<?php

namespace App\Livewire\Report;

use App\Exports\ExportStock;
use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Component;
use Maatwebsite\Excel\Facades\Excel;

class Stock extends Component
{
    use LivewireAlert;

    public $supplier;
    public $startDate;
    public $selectedDate;
    public $dataDetail;
    public $selectedWarehouse = null;
    public $selectedCodeWarehouse;
    public $selectedIndexWarehouse;
    public $searchwarehouse;
    public $warehouse;
    public $stock;

    public $whsCheckboxes = [];
    public $whsSelectAll = false;
    public $flagWhs = true;
    public $whs;

    public function mount()
    {
        $this->startDate = Carbon::now()->format('Y-m-d');
        $this->dispatch('getWhs');
    }

    public function filters()
    {
        $user = Auth::user();
        $whs = Warehouse::where('code', $user->whs_code)->where('company_code', auth()->user()->company_code)->first();

        $this->stock = DB::table('Luv2_item_trans')
            ->where('Luv2_item_trans.company_code', auth()->user()->company_code)  // Company code condition for the main table
            ->where('Luv2_item_trans.date', '<=', $this->startDate)
            ->join('Luv2_item_pricelist', function ($join) use ($user, $whs) {
                $join->on('Luv2_item_trans.item_code', '=', 'Luv2_item_pricelist.item_code')
                    ->where('Luv2_item_pricelist.company_code', '=', $user->company_code)
                    ->where('Luv2_item_pricelist.pricelist_code', '=', $whs->pricelist_code);
            })
            ->when($this->selectedWarehouse, function ($query) {
                $query->where('Luv2_item_trans.whs_code', $this->selectedWarehouse->code);
            })
            ->leftJoin('Luv2_item', 'Luv2_item_trans.item_code', '=', 'Luv2_item.code')
            ->leftJoin('Luv2_warehouse', 'Luv2_item_trans.whs_code', '=', 'Luv2_warehouse.code')
            ->leftJoin('Luv2_brand', 'Luv2_item.brand_code', '=', 'Luv2_brand.code')
            ->leftJoin('Luv2_category', 'Luv2_item.category_code', '=', 'Luv2_category.code')
            ->leftJoin('Luv2_size', 'Luv2_item.size_code', '=', 'Luv2_size.code')

            // Add company_code conditions for joined tables
            ->where('Luv2_item.company_code', auth()->user()->company_code)
            ->where('Luv2_warehouse.company_code', auth()->user()->company_code)
            ->where('Luv2_brand.company_code', auth()->user()->company_code)
            ->where('Luv2_category.company_code', auth()->user()->company_code)
            ->where('Luv2_size.company_code', auth()->user()->company_code)

            ->select(
                'Luv2_warehouse.name as whs_name',
                'Luv2_brand.name as brand',
                'Luv2_category.name as category',
                'Luv2_size.name as size',
                'Luv2_item.code',
                'Luv2_item.barcode',
                'Luv2_item.name',
                'Luv2_item_pricelist.price',
                DB::raw('SUM("Luv2_item_trans".qty) as total_qty')
            )
            ->groupBy(
                'Luv2_warehouse.name',
                'Luv2_brand.name',
                'Luv2_category.name',
                'Luv2_size.name',
                'Luv2_item.code',
                'Luv2_item.barcode',
                'Luv2_item.name',
                'Luv2_item_pricelist.price'
            )
            ->orderBy('Luv2_warehouse.name', 'ASC')
            ->orderBy('Luv2_brand.name', 'ASC')
            ->orderBy('Luv2_category.name', 'ASC')
            ->orderBy('Luv2_item.name', 'ASC')
            ->get();


        $this->dispatch('refreshjs');
    }

    public function dd()
    {
        dd($this->stock);
    }

    public function exportExcel()
    {
        if ($this->selectedWarehouse) {
            $filename = "Stock Report " . $this->selectedWarehouse->name . " - {$this->startDate}.xlsx";
        } else {
            $filename = "Stock Report - {$this->startDate}.xlsx";
        }
        $this->dispatch('refreshjs');
        return Excel::download(new ExportStock($this->stock), $filename);
    }

    public function updated()
    {
        $this->dispatch('refreshjs');
    }

    #[On('getWhs')]
    public function getWhs()
    {
        $this->whs = Warehouse::where('active', 'Y')->orderBy('name', 'asc')->get();
        $this->dispatch('refreshjs');
    }

    public function render()
    {
        $this->whs = Warehouse::where('active', 'Y')->orderBy('name', 'asc')->get();

        $resultwarehouse = [];
        $user = Auth::user();
        $whs = Warehouse::where('code', $user->whs_code)->where('company_code', auth()->user()->company_code)->first();

        if (strlen($this->searchwarehouse) >= 1) {
            $resultwarehouse = DB::table('Luv2_warehouse')->where('name', 'ILIKE', '%' . $this->searchwarehouse . '%')
                ->where('company_code', auth()->user()->company_code)
                ->limit(10)->get();
            $this->warehouse = $resultwarehouse;
        }
        $user = Auth::user();
        if ($user->role != 'Admin') {
            $warehouse = DB::table('Luv2_warehouse')->where('code', $user->whs_code)
                ->where('company_code', auth()->user()->company_code)
                ->first();

            if ($warehouse) {

                $this->searchwarehouse = $warehouse->name;
                $this->selectedWarehouse = $warehouse;
            }
        }
        $this->stock = DB::table('Luv2_item_trans')
            ->where('Luv2_item_trans.company_code', auth()->user()->company_code)
            ->where('Luv2_item_trans.date', '<=', $this->startDate)
            ->join('Luv2_item_pricelist', function ($join) use ($user, $whs) {
                $join->on('Luv2_item_trans.item_code', '=', 'Luv2_item_pricelist.item_code')
                    ->where('Luv2_item_pricelist.company_code', '=', $user->company_code)
                    ->where('Luv2_item_pricelist.pricelist_code', '=', $whs->pricelist_code);
            })
            ->when($this->selectedWarehouse, function ($query) {
                $query->where('Luv2_item_trans.whs_code', $this->selectedWarehouse->code);
            })
            ->leftJoin('Luv2_item', function ($join) {
                $join->on('Luv2_item_trans.item_code', '=', 'Luv2_item.code')
                    ->where('Luv2_item.company_code', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_warehouse', function ($join) {
                $join->on('Luv2_item_trans.whs_code', '=', 'Luv2_warehouse.code')
                    ->where('Luv2_warehouse.company_code', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_brand', function ($join) {
                $join->on('Luv2_item.brand_code', '=', 'Luv2_brand.code')
                    ->where('Luv2_brand.company_code', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_category', function ($join) {
                $join->on('Luv2_item.category_code', '=', 'Luv2_category.code')
                    ->where('Luv2_category.company_code', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_size', function ($join) {
                $join->on('Luv2_item.size_code', '=', 'Luv2_size.code')
                    ->where('Luv2_size.company_code', auth()->user()->company_code);
            })
            ->select(
                'Luv2_warehouse.name as whs_name',
                'Luv2_brand.name as brand',
                'Luv2_category.name as category',
                'Luv2_size.name as size',
                'Luv2_item.code',
                'Luv2_item.barcode',
                'Luv2_item.name',
                'Luv2_item_pricelist.price',
                DB::raw('SUM("Luv2_item_trans".qty) as total_qty')
            )
            ->groupBy(
                'Luv2_warehouse.name',
                'Luv2_brand.name',
                'Luv2_category.name',
                'Luv2_size.name',
                'Luv2_item.code',
                'Luv2_item.barcode',
                'Luv2_item.name',
                'Luv2_item_pricelist.price',
            )
            ->orderBy('Luv2_warehouse.name', 'ASC')
            ->orderBy('Luv2_brand.name', 'ASC')
            ->orderBy('Luv2_category.name', 'ASC')
            ->orderBy('Luv2_item.name', 'ASC')
            ->get();


        return view('livewire.report.stock', [
            'resultwarehouse' => $resultwarehouse,
            'stock' => $this->stock,
            // 'whs' => $whs,
        ]);
    }
    public function clearfilters()
    {
        $this->searchwarehouse = '';
        $this->selectedWarehouse = null;
        $this->startDate = Carbon::now()->format('Y-m-d');

        $this->stock = collect();
        $this->warehouse = collect();

        $this->dispatch('refreshjs');
    }

    public function selectWarehouse($index)
    {
        $this->selectedIndexWarehouse = $index;
        $this->selectedCodeWarehouse = $this->warehouse[$index]->code;

        $warehouse = DB::table('Luv2_warehouse')->where('code', $this->selectedCodeWarehouse)
            ->where('company_code', auth()->user()->company_code)
            ->first();

        if ($warehouse) {

            $this->searchwarehouse = $warehouse->name;
            $this->selectedWarehouse = $warehouse;

            $this->selectedCodeWarehouse = null;
            $this->selectedIndexWarehouse = null;

            $this->alert('info', 'Warehouse: ' . $warehouse->name, [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->dispatch('refreshDatatable');
        }

        $this->dispatch('refreshjs');
    }

    public function selectWarehouseByClick($warehouseCode)
    {

        $warehouse = DB::table('Luv2_warehouse')->where('code', $warehouseCode)
            ->where('company_code', auth()->user()->company_code)
            ->first();

        if ($warehouse) {

            $this->searchwarehouse = $warehouse->name;
            $this->selectedWarehouse = $warehouse;

            $this->alert('info', 'Warehouse: ' . $warehouse->name, [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->dispatch('refreshDatatable');
        }

        $this->dispatch('refreshjs');
    }

    public function updatedSearchwarehouse()
    {
        $this->selectedWarehouse = null;
        $this->dispatch('refreshjs');
    }

    public function updateWhsCheckboxes($i)
    {
        $this->whsCheckboxes[$i] = !$this->checkboxes[$i];
    }
}
