<?php

namespace App\Livewire\Pos\Udf;

use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;

class CreateUDF extends Component
{
    use LivewireAlert;

    public $code;
    #[Rule('required', 'string')]
    public $name;
    #[Rule('required', 'string')]
    public $field;
    public $active = 'Y';
    public $updated_by;
    public $type = 'text';
    public $udfs;
    public $edit = false;

    public function createUDF()
    {
        $this->validate();

        $company_code = Auth::user()->company_code;
        $baseCode = strtoupper(substr(preg_replace('/\s+/', '_', $this->name), 0, 10));
        $this->code = $baseCode;
        $counter = 1;

        while (DB::table('Luv2_udf')->where('code', $this->code)->exists()) {
            $this->code = $baseCode . '_' . $counter;
            $counter++;
        }

        DB::table('Luv2_udf')->insert([
            'company_code' => $company_code,
            'code' => $this->code,
            'name' => $this->name,
            'field' => $this->field,
            'type' => $this->type,
            'active' => $this->active,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
            'updated_by' => Auth::user()->id,
        ]);

        $this->alert('success', 'UDF record created successfully.');
        $this->reset(['name', 'field', 'active']);
        $this->dispatch('dataCreated');
    }


    #[On('dataCreated')]
    public function mount()
    {
        $this->udfs = DB::table('Luv2_udf')
            ->where('company_code', Auth::user()->company_code)
            ->orderBy(column: 'name')
            ->get() ?? collect();
        $this->edit = false;
    }

    #[On('changeStatusUDF')]
    public function changeStatusUDF($code)
    {
        $currentUdf = DB::table('Luv2_udf')->where('code', $code)->first();
        $newStatus = $currentUdf->active === 'Y' ? 'N' : 'Y';

        DB::table('Luv2_udf')->where('code', $code)->update(['active' => $newStatus]);
        $this->alert('success', 'Status UDF *' . $currentUdf->name . '* changed to ' . ($newStatus === 'Y' ? 'Active' : 'Inactive'));

        $this->dispatch('dataCreated');
        $this->cancelEditUDF();
    }

    public function hydrate()
    {
        $this->dispatch('refreshjs');
    }

    #[On('editUDF')]
    public function editUDF($code)
    {
        $currentUDF = DB::table('Luv2_udf')->where('code', $code)->first();
        if ($currentUDF) {
            $this->code = $currentUDF->code;
            $this->name = $currentUDF->name;
            $this->field = $currentUDF->field;
            $this->active = $currentUDF->active;
            $this->edit = true;
        }
    }

    public function saveEditUDF()
    {
        $this->validate([
            'name' => 'required|string|max:255',
            'field' => 'required|string|max:255',
            'active' => 'required|in:Y,N',
        ]);

        DB::table('Luv2_udf')->where('code', $this->code)->update([
            'name' => $this->name,
            'field' => $this->field,
            'active' => $this->active,
            'type' => $this->type,
            'updated_at' => Carbon::now(),
            'updated_by' => Auth::user()->id,
        ]);

        $this->alert('success', 'UDF record updated successfully.');
        $this->reset(['name', 'field', 'active']);
        $this->dispatch('dataCreated');
    }

    public function cancelEditUDF()
    {
        $this->code = '';
        $this->name = '';
        $this->field = '';
        $this->active = '';
        $this->edit = false;
    }

    public function render()
    {
        return view('livewire.pos.udf.create-u-d-f');
    }
}
