<?php

namespace App\Livewire\Pos\Item;

use App\Models\Cart;
use App\Models\ItemPrice;
use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Relations\Concerns\AsPivot;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Otnansirk\SnapBI\Services\BCA\Traits\HasAccessToken;

class ItemTable extends Component
{

    use LivewireAlert;

    public $itemData = [];
    public $items;
    public $promoitem;
    public $item_code = '';
    #[Rule('required', message: 'Item Code field is required.')]
    #[Rule('min:3', message: 'Item Code field must be at least 3 characters.')]
    public string $code;

    #[Rule('required', message: 'Item Code field is required.')]
    #[Rule('min:3', message: 'Item Code field must be at least 3 characters.')]
    public string $barcode;

    #[Rule('required', message: 'Item Name field is required.')]
    #[Rule('min:4', message: 'Item Name field must be at least 4 characters.')]
    public string $name = '';

    #[Rule('required', message: 'Brand Code field is required.')]
    public string $brand_code = '';

    #[Rule('required', message: 'Category Code field is required.')]
    public string $category_code = '';

    #[Rule('required', message: 'Size Code field is required.')]
    public string $size_code = '';

    #[Rule('required', message: 'Avg Price field is required.')]
    #[Rule('numeric', message: 'Avg Price field must be a number.')]
    public string $avgprice = '';

    #[Rule('required', message: 'Price field is required.')]
    #[Rule('numeric', message: 'Price field must be a number.')]
    public string $price = '';

    public $inventory = true;
    public $sales = true;
    public $purchase = true;

    public $isCreateNewBrand = false;
    public $isCreateNewCategory = false;
    public $isCreateNewSize = false;
    public $isCreateNewUnit = false;

    public $unit_label = '';
    public $name_unit = '';
    public $name_brand = '';
    public $name_size = '';
    public $name_category = '';
    public $code_brand = '';
    public $code_size = '';
    public $code_category = '';
    public $itemCategory;
    public $itemBrand;
    public $itemSize;
    public $itemUnit;
    public $unit_code = '';
    public $qtyunit = '';
    public $keterangan = [];


    public function mount($item)
    {
        ini_set('max_execution_time', '1000');
        $this->items = $item;
        $this->itemSize = DB::table('Luv2_size')->where('company_code', Auth::user()->company_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->get();
        $this->itemBrand = DB::table('Luv2_brand')->where('company_code', Auth::user()->company_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->get();
        $this->itemCategory = DB::table('Luv2_category')->where('company_code', Auth::user()->company_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->get();
        $this->itemUnit = DB::table('Luv2_unit')->where('company_code', Auth::user()->company_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->get();
    }

    public function dispatchToDetailItem($code)
    {
        $this->promoitem = DB::table('Luv2_promo')
            ->join('Luv2_promo_item', 'Luv2_promo.no', '=', 'Luv2_promo_item.no')
            ->where('Luv2_promo_item.item_code', $code)
            ->distinct('Luv2_promo.no')
            ->orderBy('Luv2_promo.no')
            ->get();
        $this->dispatch('refreshjs');
    }

    public function test()
    {
        dd($this->items);
    }

    public function clearPromo()
    {
        $this->promoitem = [];
        $this->dispatch('refreshjs');
    }
    public function updatedBrandCode()
    {
        if ($this->brand_code == 'BrandCN') {
            $this->isCreateNewBrand = true;
        } else {
            $this->isCreateNewBrand = false;
            $this->name_brand = '';
        }
    }

    public function updatedCategoryCode()
    {
        if ($this->category_code == 'CategoryCN') {
            $this->isCreateNewCategory = true;
        } else {
            $this->isCreateNewCategory = false;
            $this->name_category = '';
        }
    }

    public function updatedSizeCode()
    {
        if ($this->size_code == 'SizeCN') {
            $this->isCreateNewSize = true;
        } else {
            $this->isCreateNewSize = false;
            $this->name_size = '';
        }
    }
    public function updatedUnitCode()
    {
        if ($this->unit_code == 'UnitCN') {
            $this->isCreateNewUnit = true;
        } else {
            $this->isCreateNewUnit = false;
            if ($this->unit_code != '') {
                $unit = DB::table('Luv2_unit')->where('id', $this->unit_code)->first();
                $this->unit_label = $unit->name;
            }
            $this->name_unit = '';
        }
    }

    public function updatedNameUnit()
    {
        $this->unit_label = $this->name_unit;
    }


    #[On('editItem')]
    public function toModalEdit($code)
    {
        $this->item_code = $code;

        $user = Auth::user();
        $item = DB::table('Luv2_item')->where('company_code', $user->company_code)->where('code', $code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->first();
        $unititem = DB::table('Luv2_item_unit')->where('item_code', $item->code)->first();
        $unit = DB::table('Luv2_unit')->where('id', $unititem->id_unit)->first();
        $this->code = $item->code;
        $this->barcode = $item->barcode;
        $this->name = $item->name;
        $this->brand_code = $item->brand_code;
        $this->category_code = $item->category_code;
        $this->size_code = $item->size_code;
        $this->unit_code = $unit->id;
        $this->unit_label = $unit->name;
        $this->qtyunit = $unititem->qty;
        $this->inventory = $item->inventory === 'Y' ? true : false;
        $this->sales = $item->sales === 'Y' ? true : false;
        $this->purchase = $item->purchase === 'Y' ? true : false;
        $this->avgprice = $item->avgprice ?? 0;
        $this->price = $item->price ?? 0;

        $this->dispatch('openTheModalEdit', $code);
        $this->dispatch('refreshjs');
    }

    public function saveUpdate()
    {
        // dd($this->code_brand);
        $user = Auth::user();
        $item = DB::table('Luv2_item')->where('company_code', $user->company_code)->where('code', $this->item_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->first();

        if ($this->isCreateNewBrand) {
            DB::table('Luv2_brand')->insert([
                'company_code' => Auth::user()->company_code,
                'code' => $this->code_brand,
                'name' => $this->name_brand,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }

        if ($this->isCreateNewCategory) {
            DB::table('Luv2_category')->insert([
                'company_code' => Auth::user()->company_code,
                'code' => $this->code_category,
                'name' => $this->name_category,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }

        if ($this->isCreateNewSize) {
            $lastFourDigits = substr(Auth::user()->company_code, -4);
            $tanggal = Carbon::now()->isoFormat('YYMMDDHHmm');
            DB::table('Luv2_size')->insert([
                'company_code' => Auth::user()->company_code,
                'code' => $this->code_size,
                'name' => $this->name_size,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
                'unique_code' => $lastFourDigits . $this->code_size . $tanggal,
            ]);
        }

        if ($this->isCreateNewUnit) {
            $units = DB::table('Luv2_unit')->where('company_code', auth()->user()->company_code)
                // ->where('whs_code', auth()->user()->whs_code)
                ->get();
            $count = $units->count();
            $id = ++$count;
            DB::table('Luv2_unit')->insert([
                'id' => $id,
                // 'whs_code' => $this->whs_code,
                'company_code' => Auth::user()->company_code,
                'name' => $this->name_unit,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }

        $updated = DB::table('Luv2_item')
            ->where('company_code', $user->company_code)
            ->where('code', $this->item_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->update([
                'company_code' => Auth::user()->company_code,
                'code' => $this->code,
                'barcode' => $this->barcode,
                'name' => $this->name,
                'brand_code' => !empty($this->code_brand) ? $this->code_brand : $this->brand_code,
                'category_code' => !empty($this->code_category) ? $this->code_category : $this->category_code,
                'size_code' => !empty($this->code_size) ? $this->code_size : $this->size_code,
                'status' => 'Y',
                'inventory' => $this->inventory ? 'Y' : 'N',
                'sales' => $this->sales ? 'Y' : 'N',
                'purchase' => $this->purchase ? 'Y' : 'N',
                'active' => 'Y',
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        $unitcode = $this->unit_code == ''
            ? DB::table('Luv2_unit')->where('company_code', auth()->user()->company_code)->where('name', $this->name_unit)
            // ->where('whs_code', auth()->user()->whs_code)
            ->first()->id
            : $this->unit_code;
        DB::table('Luv2_item_unit')
            ->where('company_code', Auth::user()->company_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->where('item_code', $this->code)
            ->update([
                'id_unit' => $unitcode,
                'qty' => $this->qtyunit,
                'updated_at' => Carbon::now(),
            ]);


        if ($updated) {
            $this->alert('success', 'Data updated successfully', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);

            $this->dispatch('closeModal');
            $this->dispatch('afterEdit');
        } else {
        }
    }

    #[On('afterEdit')]
    public function afterEdit()
    {
        sleep(3);
        return redirect()->route('item');
    }

    public function render()
    {
        $user = Auth::user();
        $lostSale = Cart::where('user_code', $user->code)
            ->where('lostsale', 'Y')
            ->select('Luv2_carts.*')
            ->addSelect([
                'total_qty' => DB::table('Luv2_item_trans')
                    ->where('company_code', auth()->user()->company_code)
                    ->where('whs_code', auth()->user()->whs_code)
                    ->selectRaw('SUM(qty)')
                    ->whereColumn('Luv2_item_trans.item_code', 'Luv2_carts.code')
                    ->groupBy('Luv2_item_trans.item_code')
            ])
            ->orderBy('grouping', 'desc')
            ->orderBy('promo', 'desc')
            ->orderBy('pack', 'asc')
            ->get();

        return view('livewire.pos.item.item-table', [
            'lostSale' => $lostSale
        ]);
    }

    #[On('showInfo')]
    public function showInfo($code)
    {
        $items = DB::table('Luv2_item as i')
            ->leftJoin('Luv2_item_unit as iu', function ($join) {
                $join->on('i.code', '=', 'iu.item_code')
                    ->where('iu.company_code', '=', auth()->user()->company_code);
            })
            ->leftJoin('Luv2_unit as u', 'iu.id_unit', '=', 'u.id')
            ->leftJoin(DB::raw('(SELECT item_code, whs_code, COALESCE(SUM(qty), 0) as trans_qty 
                             FROM "Luv2_item_trans" 
                             WHERE company_code = \'' . auth()->user()->company_code . '\'
                             GROUP BY item_code, whs_code) as trans'), function ($join) {
                $join->on('i.code', '=', 'trans.item_code')
                    ->where('trans.whs_code', '=', auth()->user()->whs_code);
            })
            ->where('i.company_code', auth()->user()->company_code)
            ->where('i.code', $code)
            ->select([
                'i.code',
                'i.name',
                'iu.qty as item_unit_qty',
                'iu.pricebuy',
                'iu.pricesell',
                'u.name as unit_name',
                'trans.trans_qty',
            ])
            ->get();

        $itemData = [];
        foreach ($items as $item) {
            if (!isset($itemData['code'])) {
                $itemData = [
                    'code' => $item->code,
                    'name' => $item->name,
                    'units' => []
                ];
            }

            $item_unit_qty = $item->item_unit_qty ?? 0;

            if ($item_unit_qty > 0) {
                $total_qty = $item->trans_qty;
                $qty_unit = floor($total_qty / $item_unit_qty);
                $remaining_qty = $total_qty % $item_unit_qty;
            } else {
                $qty_unit = 0;
                $remaining_qty = $item->trans_qty;
            }

            $itemData['units'][] = [
                'pieces' => $item->trans_qty,
                'unit_name' => $item->unit_name,
                'qty_unit' => $qty_unit,
                'remaining_qty' => $remaining_qty,
                'pricebuy' => $item->pricebuy,
                'pricesell' => $item->pricesell,
            ];
        }

        $this->itemData = $itemData;
        $this->dispatch('openModalInfo');
    }

    public function closeModalInfo()
    {
        $this->dispatch('closeModalInfo');
        $this->dispatch('refreshjs');
    }

    #[On('markAsLostSale')]
    public function markAsLostSale($code)
    {
        $user = Auth::user();
        $whs = Warehouse::where('company_code', $user->company_code)->where('code', auth()->user()->whs_code)->first();
        $getpricelist = DB::table('Luv2_pricelist')->where('company_code', $whs->company_code)->first();
        $pricelist = $getpricelist->code;
        $product = DB::table('Luv2_item')->where('code', $code)
            ->where('company_code', $user->company_code)
            ->first();
        if ($product) {
            $prices = ItemPrice::where('company_code', auth()->user()->company_code)->where('item_code', $product->code)->where('pricelist_code', $pricelist)->first();
            $cartCount = Cart::where('user_code', $user->code)
            ->where('company_code', $user->company_code)
            ->count();
            $linenum = $cartCount === 0 ? 1 : $cartCount + 1;
            $cart = Cart::create([
                'company_code' => $user->company_code,
                'whs_code' => $whs->code,
                'code' => $product->code,
                'barcode' => $product->barcode,
                'name' => $product->name,
                'price' => $prices ? $prices->price : 0,
                'nprice' => $prices ? $prices->price : 0,
                'sandbox' => 'N',
                'linenum' => $linenum,
                'discount' => 0,
                'sales_code' => null,
                'user_code' => $user->code,
                'member_code' => null,
                'qty' => 1,
                'total_price' => $prices ? $prices->price : 0,
                'promo' => 0,
                'promo_tipe' => 0,
                'pack' => 0,
                'benefit' => 0,
                'grouping' => 0,
                'unit' => 0,
                'qty_unit' => 1,
                'cost' => 0,
                'consignment' => $product->consignment,
                'shared_margin' => 0,
                'lostsale' => 'Y',

            ]);
            $this->dispatch('showCanvas');
            $this->dispatch('refreshjs');
        } else {
            $this->alert('info', 'Barang tidak ditemukan', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->dispatch('refreshjs');
        }
    }

    public function deleteLostSale($id)
    {
        Cart::where('id', $id)->delete();
        $this->dispatch('afterDeleteLostSale');
    }

    #[On('afterDeleteLostSale')]
    public function afterDeleteLostSale()
    {
        $this->dispatch('closeCanvas');
        $this->dispatch('refreshjs');
    }

    public function saveLostSale()
    {
        DB::beginTransaction();

        try {
            $data = Cart::where('user_code', auth()->user()->code)
            ->where('lostsale', 'Y')
            ->select('Luv2_carts.*')
            ->addSelect([
                'total_qty' => DB::table('Luv2_item_trans')
                    ->where('company_code', auth()->user()->company_code)
                    ->where('whs_code', auth()->user()->whs_code)
                    ->selectRaw('SUM(qty)')
                    ->whereColumn('Luv2_item_trans.item_code', 'Luv2_carts.code')
                    ->groupBy('Luv2_item_trans.item_code')
            ])
            ->orderBy('grouping', 'desc')
            ->orderBy('promo', 'desc')
            ->orderBy('pack', 'asc')
            ->get();

            if ($data) {
                foreach ($data as $item) {
                    if (isset($this->keterangan[$item->id])) {
                        $item->update(['keterangan' => $this->keterangan[$item->id]]);
                    } else {
                        if ($item->total_qty > 0) {
                            $item->update(['keterangan' => 'Missing stock']);
                        } else {
                            $item->update(['keterangan' => 'Lost sale']);
                        }
                    }

                    DB::table('Luv2_lostsale')->insert([
                        'company_code' => auth()->user()->company_code,
                        'whs_code' => auth()->user()->whs_code,
                        'barcode' => $item->barcode,
                        'name' => $item->name,
                        'code' => $item->code,
                        'keterangan' => $item->keterangan,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now(),
                        'date' => Carbon::now()->format('Y-m-d')
                    ]);
                }
            }

            Cart::where('user_code', auth()->user()->code)
                ->where('lostsale', 'Y')
                ->delete();
            DB::commit();
            $this->dispatch('hideCanvas');
            $this->dispatch('refreshjs');
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

}
