<?php

namespace App\Livewire\Pos\Item;

use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Livewire\Features\SupportFileUploads\WithFileUploads;
use Maatwebsite\Excel\Facades\Excel;
use Livewire\WithPagination;
use Illuminate\Support\Facades\Log;

class InsertSize extends Component
{
    use WithFileUploads;
    use LivewireAlert;
    use WithPagination;

    #[Rule('required', message: 'Code field is required.')]
    #[Rule('min:1', message: 'Code field must be at least 1 characters.')]
    public string $code;

    #[Rule('required', message: 'Name field is required.')]
    #[Rule('min:1', message: 'Name field must be at least 1 characters.')]
    public string $name = '';

    public $file;
    public $chunks;
    public $editChunks = false;
    public $isTable = true;
    public $itemSize;
    public $warehouses;
    public $whs_code;
    public $istutorials;
    public $isEdit = false;
    public $removedDuplicates = [];

    public $page = 1;
    public $perPage = 10;
    public $search = '';
    public $sortDirection = 'ASC';
    public $sortColumn = 'name';

    protected $paginationTheme = 'bootstrap';

    public function mount()
    {
        $tutorials = DB::table('Luv2_tutorial')
            ->where('id_user', auth()->user()->id)
            ->where('menu', 'Size')
            ->first();
        if ($tutorials && $tutorials->active === 'Y') {
            $this->istutorials = true;
        } else {
            $this->istutorials = false;
        }
    }

    public function render()
    {
        $this->warehouses = Warehouse::where('company_code', auth()->user()->company_code)->get();

        if (Auth::user()->role != 'Admin') {
            $this->whs_code = Auth::user()->whs_code;
        }

        $sizes = DB::table('Luv2_size')
            ->where('company_code', Auth::user()->company_code)
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('code', 'like', '%' . $this->search . '%')
                        ->orWhere('name', 'like', '%' . $this->search . '%');
                });
            })
            ->orderBy($this->sortColumn, $this->sortDirection)
            ->paginate($this->perPage);

        return view('livewire.pos.item.insert-size', [
            'sizes' => $sizes
        ]);
    }

    public function doSort($column)
    {
        if ($this->sortColumn === $column) {
            $this->sortDirection = ($this->sortDirection === 'ASC') ? 'DESC' : 'ASC';
            return;
        }
        $this->sortColumn = $column;
        $this->sortDirection = 'ASC';

        $this->resetPage();
    }

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function importItem()
    {
        $data = $this->validate([
            'file' => 'required|file|mimes:csv,xlsx,txt',
        ]);

        try {
            $fileExtension = $this->file->getClientOriginalExtension();

            if ($fileExtension === 'csv' || $fileExtension === 'txt') {
                $this->importCSV();
            } elseif ($fileExtension === 'xlsx') {
                $this->importXLSX();
            } else {
                $this->alert('warning', 'Format file tidak didukung', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
            }

            $this->dispatch('refreshjs');
        } catch (\Throwable $th) {
            $this->alert('warning', 'Error, Format template salah!', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    private function importCSV()
    {
        try {
            $file = fopen($this->file->getRealPath(), 'r');
            if ($file === false) {
                throw new \Exception('Tidak dapat membuka file');
            }

            $skipFirstRow = true;
            $chunks = [];
            $this->removedDuplicates = [];
            $rowNumber = 0;
            $errors = [];
            $seenCodes = [];

            // Try to detect the delimiter from the first line
            $firstLine = fgets($file);
            rewind($file);

            // Count occurrences of potential delimiters
            $commaCount = substr_count($firstLine, ',');
            $tabCount = substr_count($firstLine, "\t");
            $semicolonCount = substr_count($firstLine, ';');

            // Choose the delimiter with the highest count
            $delimiter = ',';  // default
            $maxCount = $commaCount;

            if ($tabCount > $maxCount) {
                $delimiter = "\t";
                $maxCount = $tabCount;
            }
            if ($semicolonCount > $maxCount) {
                $delimiter = ";";
            }

            while (($data = fgetcsv($file, 0, $delimiter)) !== false) {
                $rowNumber++;

                if ($skipFirstRow) {
                    $skipFirstRow = false;
                    continue;
                }

                if (empty(array_filter($data))) {
                    continue;
                }

                $processedData = [
                    'code' => strval(trim($data[0])),
                    'name' => trim($data[1])
                ];

                // Validate required fields
                if (empty($processedData['code'])) {
                    $errors[] = "Row {$rowNumber}: Code is required";
                    continue;
                }
                if (empty($processedData['name'])) {
                    $errors[] = "Row {$rowNumber}: Name is required";
                    continue;
                }

                if (isset($seenCodes[$processedData['code']])) {
                    $processedData['duplicate_type'] = 'Duplikat dalam File Import';
                    $this->removedDuplicates[] = $processedData;
                    continue;
                }

                $seenCodes[$processedData['code']] = true;
                $chunks[] = $processedData;
            }

            fclose($file);

            if (!empty($errors)) {
                throw new \Exception("Ditemukan beberapa kesalahan:\n" . implode("\n", $errors));
            }

            if (empty($chunks)) {
                throw new \Exception('Tidak ada data valid dalam file CSV. Mohon periksa format template.');
            }

            $this->chunks = $chunks;

            if (!empty($this->removedDuplicates)) {
                $duplicateCount = count($this->removedDuplicates);
                $this->alert('warning', "Ditemukan $duplicateCount data duplikat yang telah dihapus. Anda dapat mengekspor data duplikat menggunakan tombol 'Export Duplikat'.", [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                    'showConfirmButton' => true,
                ]);
            }

            $this->alert('success', 'File berhasil diimpor. Silakan periksa dan klik Simpan untuk memperbarui database.', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);

            $this->dispatch('refreshjs');
        } catch (\Exception $e) {
            Log::error('Error in importCSV: ' . $e->getMessage());
            $this->alert('error', 'Error saat mengimpor CSV: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 10000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => true,
            ]);
        }
    }

    private function importXLSX()
    {
        try {
            $rows = Excel::toArray([], $this->file->getRealPath())[0];
            if (count($rows) <= 1) {
                throw new \Exception('Tidak ada data dalam file Excel.');
            }

            $chunks = [];
            $this->removedDuplicates = [];
            $skipFirstRow = true;
            $rowNumber = 0;
            $errors = [];
            $seenCodes = [];

            foreach ($rows as $row) {
                $rowNumber++;

                if ($skipFirstRow) {
                    $skipFirstRow = false;
                    continue;
                }

                if (empty(array_filter($row))) {
                    continue;
                }

                $processedData = [
                    'code' => strval(trim($row[0])),
                    'name' => trim($row[1])
                ];

                // Validate required fields
                if (empty($processedData['code'])) {
                    $errors[] = "Row {$rowNumber}: Code is required";
                    continue;
                }
                if (empty($processedData['name'])) {
                    $errors[] = "Row {$rowNumber}: Name is required";
                    continue;
                }

                if (isset($seenCodes[$processedData['code']])) {
                    $processedData['duplicate_type'] = 'Duplikat dalam File Import';
                    $this->removedDuplicates[] = $processedData;
                    continue;
                }

                $seenCodes[$processedData['code']] = true;
                $chunks[] = $processedData;
            }

            if (!empty($errors)) {
                throw new \Exception("Ditemukan beberapa kesalahan:\n" . implode("\n", $errors));
            }

            if (empty($chunks)) {
                throw new \Exception('Tidak ada data valid dalam file Excel. Mohon periksa format template.');
            }

            $this->chunks = $chunks;

            if (!empty($this->removedDuplicates)) {
                $duplicateCount = count($this->removedDuplicates);
                $this->alert('warning', "Ditemukan $duplicateCount data duplikat yang telah dihapus. Anda dapat mengekspor data duplikat menggunakan tombol 'Export Duplikat'.", [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                    'showConfirmButton' => true,
                ]);
            }

            $this->alert('success', 'File berhasil diimpor. Silakan periksa dan klik Simpan untuk memperbarui database.', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);

            $this->dispatch('refreshjs');
        } catch (\Exception $e) {
            Log::error('Error in importXLSX: ' . $e->getMessage());
            $this->alert('error', 'Error saat mengimpor Excel: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    private function checkForDuplicates($data)
    {
        try {
            // Check for duplicates within the imported data
            $codes = array_column($data, 'code');
            $duplicateCodesCount = array_count_values($codes);

            // Get list of codes that exist in database
            $existingItems = DB::table('Luv2_size')
                ->where('company_code', Auth::user()->company_code)
                ->whereIn('code', $codes)
                ->get(['code', 'name'])
                ->keyBy('code');

            // Separate data into updates and inserts
            $updatesData = [];
            $insertsData = [];
            $removedDuplicates = [];
            $seenCodes = [];

            foreach ($data as $index => $item) {
                if (!isset($item['code']) || empty($item['code'])) {
                    continue;
                }

                $code = $item['code'];

                // If code already seen in chunks, add to removed duplicates
                if (isset($seenCodes[$code])) {
                    $item['duplicate_type'] = 'Import File Duplicate';
                    $removedDuplicates[] = $item;
                    continue;
                }

                // If exists in database, add to updates and removedDuplicates
                if (isset($existingItems[$code])) {
                    $updatesData[] = $item;
                    $item['duplicate_type'] = 'Updated in Database';
                    $removedDuplicates[] = $item;
                } else {
                    // For first occurrence, add to inserts and mark as seen
                    $insertsData[] = $item;
                    $seenCodes[$code] = true;

                    // If this code appears multiple times in import, add all occurrences after first to removedDuplicates
                    if ($duplicateCodesCount[$code] > 1) {
                        foreach ($data as $dupIndex => $dupItem) {
                            if ($dupIndex > $index && $dupItem['code'] === $code) {
                                $dupItem['duplicate_type'] = 'Import File Duplicate';
                                $removedDuplicates[] = $dupItem;
                            }
                        }
                    }
                }
            }

            return [
                'hasDuplicates' => !empty($removedDuplicates),
                'updates' => $updatesData,
                'inserts' => $insertsData,
                'removedDuplicates' => $removedDuplicates,
                'existingItems' => $existingItems
            ];
        } catch (\Exception $e) {
            Log::error('Error in checkForDuplicates: ' . $e->getMessage());
            throw new \Exception('Error checking for duplicates: ' . $e->getMessage());
        }
    }

    public function getHasDuplicatesProperty()
    {
        return !empty($this->removedDuplicates);
    }

    public function downloadDuplicates($format = 'xlsx')
    {
        try {
            if (empty($this->removedDuplicates)) {
                $this->alert('warning', 'No duplicates to export', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
                return;
            }

            // Generate filename with timestamp
            $timestamp = now()->format('Y-m-d_H-i-s');
            $filename = "duplicate_sizes_{$timestamp}";

            if ($format === 'csv') {
                $filename .= '.csv';
                $headers = [
                    'Content-Type' => 'text/csv',
                    'Content-Disposition' => "attachment; filename={$filename}",
                ];

                $handle = fopen('php://temp', 'r+');

                // Add headers
                fputcsv($handle, ['Code', 'Name', 'Duplicate Type']);

                // Add data
                foreach ($this->removedDuplicates as $item) {
                    fputcsv($handle, [
                        $item['code'],
                        $item['name'],
                        $item['duplicate_type']
                    ]);
                }

                rewind($handle);
                $content = stream_get_contents($handle);
                fclose($handle);

                return response($content, 200, $headers);
            }

            // Excel export
            $filename .= '.xlsx';
            return Excel::download(new class($this->removedDuplicates) implements
                \Maatwebsite\Excel\Concerns\FromArray,
                \Maatwebsite\Excel\Concerns\WithHeadings,
                \Maatwebsite\Excel\Concerns\ShouldAutoSize
            {
                private $items;

                public function __construct($items)
                {
                    $this->items = $items;
                }

                public function array(): array
                {
                    return array_map(function ($item) {
                        return [
                            'code' => $item['code'],
                            'name' => $item['name'],
                            'duplicate_type' => $item['duplicate_type']
                        ];
                    }, $this->items);
                }

                public function headings(): array
                {
                    return ['Code', 'Name', 'Duplicate Type'];
                }
            }, $filename);
        } catch (\Exception $e) {
            $this->alert('error', 'Error exporting duplicates: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return null;
        }
    }

    public function exportCurrentTable($format = 'xlsx')
    {
        try {
            // Get current filtered/searched sizes
            $sizes = DB::table('Luv2_size')
                ->where('company_code', Auth::user()->company_code)
                ->when($this->search, function ($query) {
                    $query->where(function ($q) {
                        $q->where('code', 'like', '%' . $this->search . '%')
                            ->orWhere('name', 'like', '%' . $this->search . '%');
                    });
                })
                ->orderBy($this->sortColumn, $this->sortDirection)
                ->get();

            if ($sizes->isEmpty()) {
                $this->alert('warning', 'No data to export', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
                return;
            }

            // Transform sizes for export
            $exportData = $sizes->map(function ($size) {
                return [
                    'code' => $size->code,
                    'name' => $size->name,
                    'created_at' => $size->created_at,
                    'updated_at' => $size->updated_at,
                ];
            })->toArray();

            // Generate filename with timestamp
            $timestamp = now()->format('Y-m-d_H-i-s');
            $filename = "sizes_export_{$timestamp}";

            if ($format === 'csv') {
                $filename .= '.csv';
                $headers = [
                    'Content-Type' => 'text/csv',
                    'Content-Disposition' => "attachment; filename={$filename}",
                ];

                $handle = fopen('php://temp', 'r+');

                // Add headers
                fputcsv($handle, [
                    'Code',
                    'Name',
                    'Created At',
                    'Updated At'
                ]);

                // Add data
                foreach ($exportData as $row) {
                    fputcsv($handle, $row);
                }

                rewind($handle);
                $content = stream_get_contents($handle);
                fclose($handle);

                return response($content, 200, $headers);
            }

            // Excel export
            $filename .= '.xlsx';
            return Excel::download(new class($exportData) implements
                \Maatwebsite\Excel\Concerns\FromArray,
                \Maatwebsite\Excel\Concerns\WithHeadings,
                \Maatwebsite\Excel\Concerns\ShouldAutoSize
            {
                private $items;

                public function __construct($items)
                {
                    $this->items = $items;
                }

                public function array(): array
                {
                    return $this->items;
                }

                public function headings(): array
                {
                    return [
                        'Code',
                        'Name',
                        'Created At',
                        'Updated At'
                    ];
                }
            }, $filename);
        } catch (\Exception $e) {
            $this->alert('error', 'Error exporting data: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return null;
        }
    }

    public function saveChunks()
    {
        DB::beginTransaction();
        try {
            $userId = Auth::user()->id;

            // Get list of codes that exist in database
            $codes = array_column($this->chunks, 'code');
            $existingItems = DB::table('Luv2_size')
                ->where('company_code', Auth::user()->company_code)
                ->whereIn('code', $codes)
                ->pluck('code')
                ->toArray();

            $updateCount = 0;
            $insertCount = 0;
            $lastFourDigits = substr(Auth::user()->company_code, -4);
            $tanggal = Carbon::now()->isoFormat('YYMMDDHHmm');

            foreach ($this->chunks as $chunk) {
                $data = [
                    'company_code' => Auth::user()->company_code,
                    'code' => $chunk['code'],
                    'name' => $chunk['name'],
                    'updated_at' => Carbon::now(),
                ];

                if (in_array($chunk['code'], $existingItems)) {
                    // Update existing record
                    DB::table('Luv2_size')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('code', $chunk['code'])
                        ->update($data);
                    $updateCount++;
                } else {
                    // Insert new record
                    $data['created_at'] = Carbon::now();
                    $data['unique_code'] = $lastFourDigits . $chunk['code'] . $tanggal;
                    DB::table('Luv2_size')->insert($data);
                    $insertCount++;
                }
            }

            DB::commit();

            // Build the message
            $message = "Data berhasil disimpan ke database.";
            if ($insertCount > 0) {
                $message .= " Menambahkan {$insertCount} data baru.";
            }
            if ($updateCount > 0) {
                $message .= " Memperbarui {$updateCount} data yang sudah ada.";
            }

            $this->alert('success', $message, [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => false,
            ]);

            // Reset the import state
            $this->chunks = [];
            $this->removedDuplicates = [];
            $this->file = null;
            $this->editChunks = false;
            $this->isTable = true;

            $this->dispatch('successImport');
            $this->dispatch('refreshDatatable');
            $this->dispatch('success-import');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error saving chunks: ' . $e->getMessage());
            $this->alert('error', 'Error saat menyimpan data: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    public function changeTable()
    {
        $this->isTable = !$this->isTable;

        $this->chunks = [];
        $this->file = '';

        if (!$this->editChunks) {
            $this->dispatch('refreshjs');
        }
    }

    public function modeEdit()
    {
        $this->editChunks = !$this->editChunks;

        if (!$this->editChunks) {
            $this->dispatch('refreshjs');
        }
    }

    public function modeEditTrue()
    {
        $this->editChunks = true;
    }

    public function modeEditFalse()
    {
        $this->editChunks = false;
        $this->dispatch('refreshjs');
    }

    public function cancelImport()
    {
        $this->chunks = [];
        $this->file = '';
        $this->dispatch('refreshjs');
    }

    public function save()
    {
        $this->validate();

        $lastFourDigits = substr(Auth::user()->company_code, -4);
        $tanggal = Carbon::now()->isoFormat('YYMMDDHHmm');

        DB::table('Luv2_size')->insert([
            'company_code' => Auth::user()->company_code,
            'code' => strtoupper($this->code),
            'name' => $this->name,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
            'unique_code' => $lastFourDigits . strtoupper($this->code) . $tanggal,
        ]);

        $this->alert('success', 'Data Size berhasil disimpan', [
            'position' => 'top',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);

        $this->dispatch('closemodalCreate');
        $this->dispatch('afterSave');
    }

    #[On('closeModal')]
    public function closeModal()
    {
        $this->code = '';
        $this->name = '';

        $this->dispatch('closemodalCreate');
        $this->alert('success', 'Data Size berhasil disimpan', [
            'position' => 'top',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);
    }

    #[On('afterSave')]
    public function dataSizeafterSave()
    {
        $this->itemSize = DB::table('Luv2_size')->where('company_code', Auth::user()->company_code)
            ->get();
    }

    #[On('isEdit')]
    public function isEdit($code)
    {
        $this->isEdit = true;
        $size = DB::table('Luv2_size')->where('company_code', Auth::user()->company_code)->where('code', $code)->first();
        if ($size) {
            $this->code = $size->code;
            $this->name = $size->name;
        }
        $this->dispatch('refreshjs');
        $this->dispatch('openModalEdit', $this->code, $this->name);
    }

    public function saveUpdate()
    {
        DB::table('Luv2_size')->where('company_code', Auth::user()->company_code)->where('code', $this->code)->update([
            'name' => $this->name,
            'updated_at' => Carbon::now(),
        ]);
        $this->alert('success', 'Data Size berhasil dirubah', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);
        $this->dispatch('clearAfterCloseModalEdit');

        $this->dispatch('refreshjs');
        $this->isEdit = false;
    }

    #[On('markAsDeleted')]
    public function markAsDeleted($code)
    {
        $item = DB::table('Luv2_item')->where('company_code', Auth::user()->company_code)->where('size_code', $code)->first();
        if ($item) {
            $this->alert('warning', 'Data Size tidak dapat dihapus, karena masih digunakan pada item', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        } else {
            DB::table('Luv2_size')->where('company_code', Auth::user()->company_code)->where('code', $code)->delete();
            $this->alert('success', 'Data Size berhasil dihapus', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
        $this->dispatch('refreshjs');
    }

    public function generateNextCode()
    {
        try {
            $lastCode = DB::table('Luv2_size')
                ->where('company_code', auth()->user()->company_code)
                ->whereRaw("code ~ '^[A-Z][0-9]{2}$'") // Matches A01 to Z99
                ->orderBy('code', 'desc')
                ->first();



            if (!$lastCode) {
                // If no codes exist, start with A01
                return 'A01';
            }

            $lastCode = $lastCode->code;

            // Extract letter and number
            $letter = substr($lastCode, 0, 1);
            $number = intval(substr($lastCode, 1));

            // Validate the letter is A-Z
            if (!preg_match('/^[A-Z]$/', $letter)) {
                return 'A01';
            }

            if ($number < 99) {
                // If number is less than 99, increment the number
                return $letter . str_pad($number + 1, 2, '0', STR_PAD_LEFT);
            } else {
                // If number is 99, move to next letter and start from 01
                $nextLetter = chr(ord($letter) + 1);
                if (ord($nextLetter) <= ord('Z')) {
                    return $nextLetter . '01';
                } else {
                    throw new \Exception('No more available codes (reached Z99)');
                }
            }
        } catch (\Exception $e) {
            Log::error('Error generating size code: ' . $e->getMessage());
            throw $e;
        }
    }

    public function getNextAvailableCode()
    {
        try {
            $code = $this->generateNextCode();
            return ['success' => true, 'code' => $code];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Failed to generate code: ' . $e->getMessage()
            ];
        }
    }
}
