<?php

namespace App\Livewire\Pos\Item;

use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Livewire\Component;
use Livewire\Features\SupportFileUploads\WithFileUploads;
use Livewire\WithPagination;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Log;

class InsertPricelist extends Component
{
    use WithFileUploads;
    use LivewireAlert;
    use WithPagination;

    protected $paginationTheme = 'bootstrap';

    public $perPage = 10;
    public $search = '';
    public $sortColumn = 'name';
    public $sortDirection = 'ASC';

    protected $queryString = [
        'search' => ['except' => ''],
        'sortColumn' => ['except' => 'name'],
        'sortDirection' => ['except' => 'ASC']
    ];

    #[Rule('required', message: 'Code field is required.')]
    #[Rule('min:3', message: 'Code field must be at least 3 characters.')]
    #[Rule('unique:Luv2_pricelist,code', message: 'Code has been taken and must be unique.')]
    public string $code;


    #[Rule('required', message: 'Name field is required.')]
    #[Rule('min:4', message: 'Name field must be at least 4 characters.')]
    public string $name = '';

    public $file;
    public $chunks;
    public $editChunks = false;
    public $isTable = true;
    public $itemPricelist;
    public $whs_code;
    public $warehouses;
    public $istutorials;
    public $isEdit = false;
    public $removedDuplicates = [];

    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function doSort($column)
    {
        if ($this->sortColumn === $column) {
            $this->sortDirection = ($this->sortDirection === 'ASC') ? 'DESC' : 'ASC';
            return;
        }
        $this->sortColumn = $column;
        $this->sortDirection = 'ASC';

        $this->resetPage();
    }

    public function render()
    {
        $query = DB::table('Luv2_pricelist')
            ->where('company_code', Auth::user()->company_code);

        // Apply search filter
        if ($this->search) {
            $query->where(function($q) {
                $q->where('code', 'like', '%' . $this->search . '%')
                  ->orWhere('name', 'like', '%' . $this->search . '%');
            });
        }

        // Apply sorting
        $query->orderBy($this->sortColumn, $this->sortDirection);
        
        // Get paginated results
        $pricelists = $query->paginate($this->perPage);

        $this->warehouses = Warehouse::where('company_code', auth()->user()->company_code)->get();

        if(Auth::user()->role != 'Admin') {
            $this->whs_code = Auth::user()->whs_code;
        }

        return view('livewire.pos.item.insert-pricelist', [
            'pricelists' => $pricelists
        ]);
    }

    public function mount()
    {
        $tutorials = DB::table('Luv2_tutorial')
            ->where('id_user', auth()->user()->id)
            ->where('menu', 'Pricelist')
            ->first();
        if ($tutorials && $tutorials->active === 'Y') {
            $this->istutorials = true;
        } else {
            $this->istutorials = false;
        }

        // Initialize pricelist data
        $this->loadPricelists();
    }

    public function loadPricelists()
    {
        // This method is now only used to refresh the data
        $this->dispatch('$refresh');
    }

    #[On('isEdit')]
    public function isEdit($code)
    {
        $this->isEdit = true;
        $pricelist = DB::table('Luv2_pricelist')->where('company_code', Auth::user()->company_code)->where('code', $code)->first();
        if ($pricelist) {
            $this->code = $pricelist->code;
            $this->name = $pricelist->name;
            $this->dispatch('openModalEdit', $this->code, $this->name);
        }
    }

    public function save()
    {
        $checkPricelist = DB::table('Luv2_pricelist')->where('company_code', Auth::user()->company_code)->where('code', $this->code)->first();
        if ($checkPricelist) {
            $this->alert('error', 'Pricelist already exist', [
                'position' => 'top',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->addError('code', 'Code pricelist sudah digunakan');
            return;
        }
        DB::table('Luv2_pricelist')->insert([
            'company_code' => Auth::user()->company_code,
            'code' => $this->code,
            'name' => $this->name,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
        ]);

        $this->alert('success', 'Pricelist created successfully', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true
        ]);
        
        $this->resetForm();
        $this->js("$('#modalCreate').modal('hide')");
        $this->dispatch('refreshData');
    }

    public function saveUpdate()
    {
        DB::table('Luv2_pricelist')->where('company_code', Auth::user()->company_code)->where('code', $this->code)->update([
            'name' => $this->name,
            'updated_at' => Carbon::now(),
        ]);
        
        $this->alert('success', 'Pricelist updated successfully', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);
        
        $this->resetForm();
        $this->js("$('#modalCreate').modal('hide')");
        $this->dispatch('refreshData');
        $this->isEdit = false;
    }

    private function resetForm()
    {
        $this->reset(['code', 'name', 'isEdit']);
    }

    public function importItem()
    {
        $data = $this->validate([
            'file' => 'required|file|mimes:csv,xlsx,txt',
        ]);

        try {
            $fileExtension = $this->file->getClientOriginalExtension(); 

            if ($fileExtension === 'csv' || $fileExtension === 'txt') {
                $this->importCSV();
            } elseif ($fileExtension === 'xlsx') {
                $this->importXLSX();
            } else {
                $this->alert('warning', 'Format file tidak didukung', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
            }

            $this->dispatch('refreshjs');
        } catch (\Throwable $th) {
            $this->alert('warning', 'Error, Format template salah!', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    private function importCSV()
    {
        try {
            $file = fopen($this->file->getRealPath(), 'r');
            if ($file === false) {
                throw new \Exception('Tidak dapat membuka file');
            }

            $skipFirstRow = true;
            $chunks = [];
            $this->removedDuplicates = [];
            $rowNumber = 0;
            $errors = [];
            $seenCodes = [];

            // Try to detect the delimiter from the first line
            $firstLine = fgets($file);
            rewind($file);
            
            // Count occurrences of potential delimiters
            $commaCount = substr_count($firstLine, ',');
            $tabCount = substr_count($firstLine, "\t");
            $semicolonCount = substr_count($firstLine, ';');
            
            // Choose the delimiter with the highest count
            $delimiter = ',';  // default
            $maxCount = $commaCount;
            
            if ($tabCount > $maxCount) {
                $delimiter = "\t";
                $maxCount = $tabCount;
            }
            if ($semicolonCount > $maxCount) {
                $delimiter = ";";
            }

            while (($data = fgetcsv($file, 0, $delimiter)) !== false) {
                $rowNumber++;

                if ($skipFirstRow) {
                    $skipFirstRow = false;
                    continue;
                }

                if (empty(array_filter($data))) {
                    continue;
                }

                $processedData = [
                    'code' => strval(trim($data[0])),
                    'name' => trim($data[1])
                ];

                // Validate required fields
                if (empty($processedData['code'])) {
                    $errors[] = "Row {$rowNumber}: Code is required";
                    continue;
                }
                if (empty($processedData['name'])) {
                    $errors[] = "Row {$rowNumber}: Name is required";
                    continue;
                }

                if (isset($seenCodes[$processedData['code']])) {
                    $processedData['duplicate_type'] = 'Duplikat dalam File Import';
                    $this->removedDuplicates[] = $processedData;
                    continue;
                }
                
                $seenCodes[$processedData['code']] = true;
                $chunks[] = $processedData;
            }
            
            fclose($file);

            if (!empty($errors)) {
                throw new \Exception("Ditemukan beberapa kesalahan:\n" . implode("\n", $errors));
            }

            if (empty($chunks)) {
                throw new \Exception('Tidak ada data valid dalam file CSV. Mohon periksa format template.');
            }

            $this->chunks = $chunks;

            if (!empty($this->removedDuplicates)) {
                $duplicateCount = count($this->removedDuplicates);
                $this->alert('warning', "Ditemukan $duplicateCount data duplikat yang telah dihapus. Anda dapat mengekspor data duplikat menggunakan tombol 'Export Duplikat'.", [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                    'showConfirmButton' => true,
                ]);
            }

            $this->alert('success', 'File berhasil diimpor. Silakan periksa dan klik Simpan untuk memperbarui database.', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            
            $this->dispatch('refreshjs');
            
        } catch (\Exception $e) {
            $this->alert('error', 'Error saat mengimpor CSV: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 10000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => true,
            ]);
        }
    }

    private function importXLSX()
    {
        try {
            $rows = Excel::toArray([], $this->file->getRealPath())[0];
            if (count($rows) <= 1) {
                throw new \Exception('Tidak ada data dalam file Excel.');
            }

            $chunks = [];
            $this->removedDuplicates = [];
            $skipFirstRow = true;
            $rowNumber = 0;
            $errors = [];
            $seenCodes = [];

            foreach ($rows as $row) {
                $rowNumber++;

                if ($skipFirstRow) {
                    $skipFirstRow = false;
                    continue;
                }

                if (empty(array_filter($row))) {
                    continue;
                }

                $processedData = [
                    'code' => strval(trim($row[0])),
                    'name' => trim($row[1])
                ];

                // Validate required fields
                if (empty($processedData['code'])) {
                    $errors[] = "Row {$rowNumber}: Code is required";
                    continue;
                }
                if (empty($processedData['name'])) {
                    $errors[] = "Row {$rowNumber}: Name is required";
                    continue;
                }

                if (isset($seenCodes[$processedData['code']])) {
                    $processedData['duplicate_type'] = 'Duplikat dalam File Import';
                    $this->removedDuplicates[] = $processedData;
                    continue;
                }
                
                $seenCodes[$processedData['code']] = true;
                $chunks[] = $processedData;
            }

            if (!empty($errors)) {
                throw new \Exception("Ditemukan beberapa kesalahan:\n" . implode("\n", $errors));
            }

            if (empty($chunks)) {
                throw new \Exception('Tidak ada data valid dalam file Excel. Mohon periksa format template.');
            }

            $this->chunks = $chunks;

            if (!empty($this->removedDuplicates)) {
                $duplicateCount = count($this->removedDuplicates);
                $this->alert('warning', "Ditemukan $duplicateCount data duplikat yang telah dihapus. Anda dapat mengekspor data duplikat menggunakan tombol 'Export Duplikat'.", [
                    'position' => 'top-end',
                    'timer' => 5000,
                    'toast' => true,
                    'timerProgressBar' => true,
                    'showConfirmButton' => true,
                ]);
            }

            $this->alert('success', 'File berhasil diimpor. Silakan periksa dan klik Simpan untuk memperbarui database.', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);

            $this->dispatch('refreshjs');

        } catch (\Exception $e) {
            $this->alert('error', 'Error saat mengimpor Excel: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    private function checkForDuplicates($data)
    {
        try {
            // Check for duplicates within the imported data
            $codes = array_column($data, 'code');
            $duplicateCodesCount = array_count_values($codes);
            
            // Get list of codes that exist in database
            $existingItems = DB::table('Luv2_pricelist')
                ->where('company_code', Auth::user()->company_code)
                ->whereIn('code', $codes)
                ->get(['code', 'name'])
                ->keyBy('code');

            // Separate data into updates and inserts
            $updatesData = [];
            $insertsData = [];
            $removedDuplicates = [];
            $seenCodes = [];

            foreach ($data as $index => $item) {
                if (!isset($item['code']) || empty($item['code'])) {
                    continue;
                }

                $code = $item['code'];
                
                // If code already seen in chunks, add to removed duplicates
                if (isset($seenCodes[$code])) {
                    $item['duplicate_type'] = 'Import File Duplicate';
                    $removedDuplicates[] = $item;
                    continue;
                }
                
                // If exists in database, add to updates and removedDuplicates
                if (isset($existingItems[$code])) {
                    $updatesData[] = $item;
                    $item['duplicate_type'] = 'Updated in Database';
                    $removedDuplicates[] = $item;
                } else {
                    // For first occurrence, add to inserts and mark as seen
                    $insertsData[] = $item;
                    $seenCodes[$code] = true;

                    // If this code appears multiple times in import, add all occurrences after first to removedDuplicates
                    if ($duplicateCodesCount[$code] > 1) {
                        foreach ($data as $dupIndex => $dupItem) {
                            if ($dupIndex > $index && $dupItem['code'] === $code) {
                                $dupItem['duplicate_type'] = 'Import File Duplicate';
                                $removedDuplicates[] = $dupItem;
                            }
                        }
                    }
                }
            }

            return [
                'hasDuplicates' => !empty($removedDuplicates),
                'updates' => $updatesData,
                'inserts' => $insertsData,
                'removedDuplicates' => $removedDuplicates,
                'existingItems' => $existingItems
            ];
        } catch (\Exception $e) {
            Log::error('Error in checkForDuplicates: ' . $e->getMessage());
            throw new \Exception('Error checking for duplicates: ' . $e->getMessage());
        }
    }

    private function processUpdates($updates)
    {
        try {
            foreach ($updates as $item) {
                if (!isset($item['code']) || empty($item['code'])) {
                    continue;
                }

                DB::table('Luv2_pricelist')
                    ->where('company_code', Auth::user()->company_code)
                    ->where('code', $item['code'])
                    ->update([
                        'name' => $item['name'] ?? '',
                        'updated_at' => Carbon::now()
                    ]);
            }
        } catch (\Exception $e) {
            Log::error('Error in processUpdates: ' . $e->getMessage());
            throw new \Exception('Error updating records: ' . $e->getMessage());
        }
    }

    public function saveChunks()
    {
        DB::beginTransaction();
        try {
            $userId = Auth::user()->id;
            
            // Get list of codes that exist in database
            $codes = array_column($this->chunks, 'code');
            $existingItems = DB::table('Luv2_pricelist')
                ->where('company_code', Auth::user()->company_code)
                ->whereIn('code', $codes)
                ->pluck('code')
                ->toArray();

            $updateCount = 0;
            $insertCount = 0;

            foreach ($this->chunks as $chunk) {
                $data = [
                    'company_code' => Auth::user()->company_code,
                    'code' => $chunk['code'],
                    'name' => $chunk['name'],
                    'updated_at' => Carbon::now(),
                ];

                if (in_array($chunk['code'], $existingItems)) {
                    // Update existing record
                    DB::table('Luv2_pricelist')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('code', $chunk['code'])
                        ->update($data);
                    $updateCount++;
                } else {
                    // Insert new record
                    $data['created_at'] = Carbon::now();
                    DB::table('Luv2_pricelist')->insert($data);
                    $insertCount++;
                }
            }

            DB::commit();

            $message = "Data berhasil disimpan ke database. ";
            if ($insertCount > 0) {
                $message .= "Menambahkan $insertCount data baru. ";
            }
            if ($updateCount > 0) {
                $message .= "Memperbarui $updateCount data yang sudah ada.";
            }

            $this->alert('success', $message, [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => false,
            ]);

            // Reset the import state
            $this->chunks = [];
            $this->removedDuplicates = [];
            $this->file = null;
            $this->editChunks = false;
            $this->isTable = true;

            $this->dispatch('successImport');
            $this->dispatch('refreshDatatable');
            $this->dispatch('success-import'); // Dispatch browser event to hide import section

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error saving chunks: ' . $e->getMessage());
            $this->alert('error', 'Error saat menyimpan data: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    public function downloadDuplicates($format = 'xlsx')
    {
        try {
            if (empty($this->removedDuplicates)) {
                $this->alert('error', 'Tidak ada data duplikat yang tersedia untuk diekspor', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                    'showConfirmButton' => false,
                ]);
                return null;
            }

            // Generate filename with timestamp
            $timestamp = now()->format('Y-m-d_H-i-s');
            $filename = "duplicate_pricelists_{$timestamp}";

            if ($format === 'csv') {
                $filename .= '.csv';
                $headers = [
                    'Content-Type' => 'text/csv',
                    'Content-Disposition' => "attachment; filename={$filename}",
                ];

                $handle = fopen('php://temp', 'r+');
                
                // Add headers
                fputcsv($handle, ['Code', 'Name', 'Duplicate Type']);
                
                // Add data
                foreach ($this->removedDuplicates as $item) {
                    fputcsv($handle, [
                        $item['code'] ?? '',
                        $item['name'] ?? '',
                        $item['duplicate_type'] ?? 'Import File Duplicate'
                    ]);
                }

                rewind($handle);
                $content = stream_get_contents($handle);
                fclose($handle);

                return response($content, 200, $headers);
            } 
            
            // Excel export
            $filename .= '.xlsx';
            return Excel::download(new class($this->removedDuplicates) implements 
                \Maatwebsite\Excel\Concerns\FromArray,
                \Maatwebsite\Excel\Concerns\WithHeadings,
                \Maatwebsite\Excel\Concerns\ShouldAutoSize
            {
                private $items;

                public function __construct($items)
                {
                    $this->items = $items;
                }

                public function array(): array
                {
                    return $this->items;
                }

                public function headings(): array
                {
                    return ['Code', 'Name', 'Duplicate Type'];
                }
            }, $filename);

        } catch (\Exception $e) {
            Log::error('Error in downloadDuplicates: ' . $e->getMessage());
            $this->alert('error', 'Error saat mengunduh data duplikat: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => false,
            ]);
            return null;
        }
    }

    public function getHasDuplicatesProperty()
    {
        return !empty($this->removedDuplicates);
    }

    public function updateChunk($index, $field, $value)
    {
        $this->chunks[$index][$field] = $value;
    }

    public function updatedCode()
    {
        $this->resetErrorBag('code');
    }

    public function modeEdit()
    {
        $this->editChunks = !$this->editChunks;

        if (!$this->editChunks) {
            $this->dispatch('refreshjs');
        }
    }

    public function edit($code)
    {
        $this->isEdit = true;
        $pricelist = DB::table('Luv2_pricelist')
            ->where('company_code', Auth::user()->company_code)
            ->where('code', $code)
            ->first();

        $this->code = $pricelist->code;
        $this->name = $pricelist->name;

        $this->dispatch('openModal');
    }

    public function update()
    {
        $this->validate([
            'name' => 'required|min:4'
        ]);

        DB::table('Luv2_pricelist')
            ->where('company_code', Auth::user()->company_code)
            ->where('code', $this->code)
            ->update([
                'name' => $this->name,
                'updated_at' => Carbon::now()
            ]);

        $this->dispatch('closeModal');
    }

    public function delete($code)
    {
        // Check if pricelist is being used
        $isUsed = DB::table('Luv2_item_pricelist')
            ->where('company_code', Auth::user()->company_code)
            ->where('pricelist_code', $code)
            ->exists();

        if ($isUsed) {
            $this->alert('error', 'Cannot delete pricelist. It is being used by items.', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true
            ]);
            return;
        }

        DB::table('Luv2_pricelist')
            ->where('company_code', Auth::user()->company_code)
            ->where('code', $code)
            ->delete();

        $this->alert('success', 'Pricelist deleted successfully', [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true
        ]);
    }

    public function changeTable()
    {
        $this->isTable = !$this->isTable;

        $this->chunks = [];
        $this->file = '';

        if (!$this->editChunks) {
            $this->dispatch('refreshjs');
        }
    }

    #[On('afterSave')]
    public function dataPricelistafterSave()
    {
        // Remove this method as it's no longer needed
        $this->dispatch('refreshData');
    }

    #[On('markAsDeleted')]
    public function deleteItem($code)
    {
        $itempricelist = DB::table('Luv2_item_pricelist')->where('company_code', Auth::user()->company_code)->where('pricelist_code', $code)->first();
        if ($itempricelist) {
            $this->alert('warning', 'Pricelist tidak bisa dihapus', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        } else {
            DB::table('Luv2_pricelist')
                ->where('company_code', auth()->user()->company_code)
                ->where('code', $code)
                ->delete();

            $this->alert('success', 'Pricelist berhasil dihapus', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
        $this->dispatch('refreshjs');
    }

    public function modeEditTrue()
    {
        $this->editChunks = true;
    }

    public function modeEditFalse()
    {
        $this->editChunks = false;
        $this->dispatch('refreshjs');
    }

    public function cancelImport()
    {
        $this->chunks = [];
        $this->removedDuplicates = [];
        $this->file = null;
        $this->editChunks = false;
        $this->dispatch('refreshjs');
    }

    #[On('refreshjs')]
    public function refreshjs()
    {
        $this->dispatch('refreshDatatable');
    }

    #[On('refreshData')]
    public function refreshData()
    {
        $this->loadPricelists();
    }

    public function gotoPage($page)
    {
        $this->setPage($page);
    }

    public function previousPage()
    {
        $this->previousPage();
    }

    public function nextPage()
    {
        $this->nextPage();
    }
}
