<?php

namespace App\Livewire\Pos\Item;

use App\Models\Item;
use App\Models\ItemTemp;
use App\Models\Warehouse;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Attributes\Rule;
use Illuminate\Validation\Rule as ValRule;

use Livewire\Component;
use Livewire\Features\SupportFileUploads\WithFileUploads;
use Maatwebsite\Excel\Facades\Excel;

class InsertItem extends Component
{
    use WithFileUploads;
    use LivewireAlert;

    public $inventory = true;
    public $sales = true;
    public $purchase = true;
    public $isConsignment = false;
    public $itemCategory;
    public $itemBrand;
    public $itemSize;
    public $itemUnit;
    public $item;

    public $isEdit = false;

    #[Rule('required', message: 'Item Code field is required.')]
    #[Rule('min:3', message: 'Item Code field must be at least 3 characters.')]
    #[Rule('unique:Luv2_item,code', message: 'Item Code has been taken and must be unique.')]
    public string $code = '';

    #[Rule('required', message: 'Item Barcode field is required.')]
    #[Rule('min:3', message: 'Item Barcode field must be at least 3 characters.')]
    #[Rule('unique:Luv2_item,barcode', message: 'Item Barcode has been taken and must be unique.')]
    public ?string $barcode = null; // Now it's okay if it's null initially

    #[Rule('required', message: 'Item Name field is required.')]
    #[Rule('min:4', message: 'Item Name field must be at least 4 characters.')]
    public string $name = '';

    #[Rule('required', message: 'Brand Code field is required.')]
    public string $brand_code = '';
    #[Rule('required', message: 'Category Code field is required.')]

    public string $category_code = '';
    #[Rule('required', message: 'Size Code field is required.')]

    public string $size_code = '';
    // #[Rule('required', message: 'Unit field is required.')]

    public string $unit_code = '';
    public $file;
    public $chunks;
    public $editChunks = false;
    public $isTable = true;
    public $isCreateNewBrand = false;
    public $isCreateNewCategory = false;
    public $isCreateNewSize = false;
    public $isCreateNewUnit = false;
    public $isCreateNewSupplier = false;
    public $name_brand = '';
    public $name_unit = '';
    public $unit_label = '';
    public $name_size = '';
    #[Rule('unique:Luv2_category,code', message: 'Category code has been taken and must be unique.')]
    public $name_category = '';
    #[Rule('unique:Luv2_brand,code', message: 'Brand code has been taken and must be unique.')]
    public $code_brand = '';
    #[Rule('unique:Luv2_size,code', message: 'Size code has been taken and must be unique.')]
    public $code_size = '';
    #[Rule('unique:Luv2_category,code', message: 'Category code has been taken and must be unique.')]
    public $code_category = '';
    public $warehouses;
    public $whs_code;
    public $qtyunit;

    public $suppliers;

    public $istutorials;

    public $canEditSupplier = true;
    public $supplier_code = '';
    public $infosupplier_name = '';
    public $infosupplier_code = '';
    public $infoitem_name = '';
    public $infoitem_code = '';
    public $infoitem_barcode = '';
    public $suppliernew_name = '';
    public $suppliernew_address = '';
    public $suppliernew_phone = '';
    public $suppliernew_email = '';
    public $shared_margin = 0;
    public $activeUDFs;
    public $udfFields = [];

    public function rules()
    {
        return [
            'code' => [
                'required',
                'min:3',
                ValRule::unique('Luv2_item')->ignore($this->code, 'code'),
            ],
            'barcode' => [
                'required',
                'min:3',
                ValRule::unique('Luv2_item')->ignore($this->barcode, 'barcode'),
            ],
            'name' => [
                'required',
                'min:4',
            ],
            'brand_code' => [
                'required',
            ],
            'category_code' => [
                'required',
            ],
            'size_code' => [
                'required',
            ],
        ];
    }

    public function render()
    {
        $this->warehouses = Warehouse::where('company_code', auth()->user()->company_code)->get();

        return view('livewire.pos.item.insert-item');
    }

    private function exportDuplicates($duplicateInfo, $chunks, $fileType)
    {
        // Collect all duplicate items
        $duplicateItems = [];
        
        // Get items with duplicate codes in import file
        foreach ($duplicateInfo['importDuplicates'] as $code) {
            foreach ($chunks as $item) {
                if ($item['code'] === $code) {
                    $duplicateItems[] = [
                        'code' => $item['code'],
                        'barcode' => $item['barcode'],
                        'name' => $item['name'],
                        'brand_code' => $item['brand_code'],
                        'category_code' => $item['category_code'],
                        'size_code' => $item['size_code'],
                        'duplicate_type' => 'Import File Duplicate'
                    ];
                }
            }
        }

        // Get existing items from database
        if (!empty($duplicateInfo['databaseDuplicates'])) {
            $existingItems = DB::table('Luv2_item')
                ->where('company_code', Auth::user()->company_code)
                ->whereIn('code', $duplicateInfo['databaseDuplicates'])
                ->get();

            foreach ($existingItems as $item) {
                $duplicateItems[] = [
                    'code' => $item->code,
                    'barcode' => $item->barcode,
                    'name' => $item->name,
                    'brand_code' => $item->brand_code,
                    'category_code' => $item->category_code,
                    'size_code' => $item->size_code,
                    'duplicate_type' => 'Existing in Database'
                ];
            }
        }

        // Generate filename with timestamp
        $timestamp = now()->format('Y-m-d_H-i-s');
        $filename = "duplicate_items_{$timestamp}";

        if ($fileType === 'csv') {
            $filename .= '.csv';
            $headers = [
                'Content-Type' => 'text/csv',
                'Content-Disposition' => "attachment; filename={$filename}",
            ];

            $handle = fopen('php://temp', 'r+');
            
            // Add headers
            fputcsv($handle, ['Code', 'Barcode', 'Name', 'Brand Code', 'Category Code', 'Size Code', 'Duplicate Type']);
            
            // Add data
            foreach ($duplicateItems as $item) {
                fputcsv($handle, $item);
            }

            rewind($handle);
            $content = stream_get_contents($handle);
            fclose($handle);

            return response($content, 200, $headers);
        } else {
            $filename .= '.xlsx';
            return Excel::download(new class($duplicateItems) implements \Maatwebsite\Excel\Concerns\FromArray, \Maatwebsite\Excel\Concerns\WithHeadings {
                private $items;

                public function __construct($items)
                {
                    $this->items = $items;
                }

                public function array(): array
                {
                    return $this->items;
                }

                public function headings(): array
                {
                    return ['Code', 'Barcode', 'Name', 'Brand Code', 'Category Code', 'Size Code', 'Duplicate Type'];
                }
            }, $filename);
        }
    }

    private function checkForDuplicates($data)
    {
        // Check for duplicates within the imported data
        $codes = array_column($data, 'code');
        $duplicateCodes = array_filter(array_count_values($codes), function($count) {
            return $count > 1;
        });

        // Check for duplicates in the database
        $existingCodes = DB::table('Luv2_item')
            ->where('company_code', Auth::user()->company_code)
            ->whereIn('code', $codes)
            ->pluck('code')
            ->toArray();

        $duplicateInfo = [
            'hasDuplicates' => !empty($duplicateCodes) || !empty($existingCodes),
            'importDuplicates' => array_keys($duplicateCodes),
            'databaseDuplicates' => $existingCodes
        ];

        return $duplicateInfo;
    }

    public function downloadDuplicates($format = 'xlsx')
    {
        if (empty($this->chunks)) {
            $this->alert('warning', 'No duplicate data available for export', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return;
        }

        $duplicateInfo = $this->checkForDuplicates($this->chunks);
        
        if (!$duplicateInfo['hasDuplicates']) {
            $this->alert('info', 'No duplicates found to export', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return;
        }

        return $this->exportDuplicates($duplicateInfo, $this->chunks, $format);
    }

    public function toImportItem()
    {
        $this->isTable = false;
        $this->dispatch('goToimportItem');
    }

    #[On('goToimportItem')]
    public function importItem()
    {
        $data = $this->validate([
            'file' => 'required|file|mimes:csv,xlsx,txt',
        ]);

        try {
            $fileExtension = $this->file->getClientOriginalExtension();

            if ($fileExtension === 'csv' || $fileExtension === 'txt') {
                $this->importCSV();
            } elseif ($fileExtension === 'xlsx') {
                $this->importXLSX();
            } else {
                $this->alert('warning', 'Format file tidak didukung', [
                    'position' => 'top-end',
                    'timer' => 3000,
                    'toast' => true,
                    'timerProgressBar' => true,
                ]);
            }
            $this->dispatch('refreshjs');
        } catch (\Throwable $th) {
            $this->alert('warning', 'Error, check the template!', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    public function importCSV()
    {
        $file = fopen($this->file->getRealPath(), 'r');
        $skipFirstRow = true;
        $chunks = [];

        while (($data = fgetcsv($file)) !== false) {
            if ($skipFirstRow) {
                $skipFirstRow = false;
                continue;
            }
            $chunks[] = [
                'company_code' => Auth::user()->company_code,
                'code' => strval($data[0]),
                'barcode' => $data[1],
                'name' => $data[2],
                'brand_code' => $data[3],
                'category_code' => $data[4],
                'size_code' => $data[5],
                'status' => 'Y',
                'inventory' => isset($data[6]) ? $data[6] : 'Y',
                'sales' => isset($data[7]) ? $data[7] : 'Y',
                'purchase' => isset($data[8]) ? $data[8] : 'Y',
                'consignment' => isset($data[9]) ? $data[9] : 'Y',
                'supplier_code' => isset($data[10]) ? $data[10] : null,
                'active' => 'Y',
                'price' => isset($data[11]) && is_numeric($data[11]) ? $data[11] : 0,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
                'id_user' => Auth::user()->id,
            ];
        }
        fclose($file);

        // Check for duplicates
        $duplicateInfo = $this->checkForDuplicates($chunks);
        if ($duplicateInfo['hasDuplicates']) {
            $message = '';
            if (!empty($duplicateInfo['importDuplicates'])) {
                $message .= 'Duplicate codes in import file: ' . implode(', ', $duplicateInfo['importDuplicates']) . '. ';
            }
            if (!empty($duplicateInfo['databaseDuplicates'])) {
                $message .= 'Codes already exist in database: ' . implode(', ', $duplicateInfo['databaseDuplicates']);
            }
            
            $this->alert('warning', $message . ' Click to download duplicates list.', [
                'position' => 'top-end',
                'timer' => 7000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => true,
                'confirmButtonText' => 'Download XLSX',
                'onConfirmed' => 'downloadDuplicates'
            ]);

            $this->chunks = $chunks; // Store chunks for potential export
            return;
        }

        $this->chunks = $chunks;
    }

    public function importXLSX()
    {
        $rows = Excel::toArray([], $this->file->getRealPath())[0];
        $chunks = [];
        $skipFirstRow = true;
        $index = 0;

        foreach ($rows as $row) {
            if ($skipFirstRow) {
                $skipFirstRow = false;
                continue;
            }

            $chunks[] = [
                'company_code' => Auth::user()->company_code,
                'code' => strval($row[0]),
                'barcode' => $row[1],
                'name' => $row[2],
                'brand_code' => $row[3],
                'category_code' => $row[4],
                'size_code' => $row[5],
                'status' => 'Y',
                'inventory' => isset($row[6]) ? $row[6] : 'Y',
                'sales' => isset($row[7]) ? $row[7] : 'Y',
                'purchase' => isset($row[8]) ? $row[8] : 'Y',
                'consignment' => isset($row[9]) ? $row[9] : 'Y',
                'supplier_code' => isset($row[10]) ? $row[10] : null,
                'active' => 'Y',
                'price' => isset($row[11]) && is_numeric($row[11]) ? $row[11] : 0,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
                'id_user' => Auth::user()->id,
            ];

            $index++;
        }

        // Check for duplicates
        $duplicateInfo = $this->checkForDuplicates($chunks);
        if ($duplicateInfo['hasDuplicates']) {
            $message = '';
            if (!empty($duplicateInfo['importDuplicates'])) {
                $message .= 'Duplicate codes in import file: ' . implode(', ', $duplicateInfo['importDuplicates']) . '. ';
            }
            if (!empty($duplicateInfo['databaseDuplicates'])) {
                $message .= 'Codes already exist in database: ' . implode(', ', $duplicateInfo['databaseDuplicates']);
            }
            
            $this->alert('warning', $message . ' Click to download duplicates list.', [
                'position' => 'top-end',
                'timer' => 7000,
                'toast' => true,
                'timerProgressBar' => true,
                'showConfirmButton' => true,
                'confirmButtonText' => 'Download XLSX',
                'onConfirmed' => 'downloadDuplicates'
            ]);

            $this->chunks = $chunks; // Store chunks for potential export
            return;
        }

        $this->chunks = $chunks;
    }
    public function changeTable()
    {
        $this->isTable = !$this->isTable;

        if ($this->isTable) {
            $this->dispatch('reGet');
        }

        $this->chunks = [];
        $this->file = '';

        if (!$this->editChunks) {
            $this->dispatch('refreshjs');
        }
    }

    #[On('refreshjs')]
    public function refreshjs()
    {
        $this->dispatch('refreshDatatable');
    }

    public function cancelImport()
    {
        $this->dispatch('cancelImport');
        $this->chunks = [];
    }
    private function hasConsignmentWithoutSupplier()
    {
        foreach ($this->chunks as $chunk) {
            if ($chunk['consignment'] === 'Y' && empty($chunk['supplier_code'])) {
                return true;
            }
        }
        return false;
    }
    public function saveChunks()
    {
        if ($this->hasConsignmentWithoutSupplier()) {
            $this->alert('error', 'Beberapa item adalah konsinyasi tetapi tidak memiliki supplier.', [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return;
        }

        try {

            $userId = Auth::user()->id;
            ItemTemp::where('id_user', $userId)->delete();
            $chunksItem = array_chunk($this->chunks, 1000);
            foreach ($chunksItem as $chunk) {
                $insertData = [];

                foreach ($chunk as $data) {
                    $insertData[] = [
                        'company_code' => Auth::user()->company_code,
                        'code' => $data['code'] ?? null,
                        'barcode' => $data['barcode'] ?? null,
                        'name' => $data['name'] ?? null,
                        'brand_code' => $data['brand_code'] ?? null,
                        'category_code' => $data['category_code'] ?? null,
                        'size_code' => $data['size_code'] ?? null,
                        'status' => 'Y',
                        'inventory' => !empty($data['inventory']) ? (string) $data['inventory'] : 'Y',
                        'sales' => !empty($data['sales']) ? (string) $data['sales'] : 'Y',
                        'purchase' => !empty($data['purchase']) ? (string) $data['purchase'] : 'Y',
                        'active' => !empty($data['active']) ? (string) $data['active'] : 'Y',
                        'avgprice' => 0,
                        'price' => is_numeric($data['price']) ? $data['price'] : 0,
                        'consignment' => !empty($data['consignment']) ? (string) $data['consignment'] : 'Y',
                        'supp_code' => !empty($data['supp_code']) ? (string) $data['supp_code'] : null,
                        'created_at' => $data['created_at'] ?? Carbon::now(),
                        'updated_at' => $data['updated_at'] ?? Carbon::now(),
                        'id_user' => Auth::user()->id,
                    ];
                }
                DB::table('Luv2_item_temp')->insert($insertData);
            }
            $this->chunks = [];
            $this->file = '';

            DB::statement("
            INSERT INTO \"Luv2_item\" (\"company_code\", \"code\", \"barcode\", \"name\", \"brand_code\", \"category_code\", \"size_code\", \"status\", \"inventory\", \"sales\", \"purchase\", \"active\", \"avgprice\" , \"price\", \"consignment\", \"supp_code\", \"created_at\", \"updated_at\") 
            SELECT DISTINCT \"company_code\", \"code\", \"barcode\", \"name\", \"brand_code\", \"category_code\", \"size_code\", \"status\", \"inventory\", \"sales\", \"purchase\", \"active\", \"avgprice\" ,\"price\", \"consignment\", \"supp_code\", \"created_at\", \"updated_at\"
            FROM \"Luv2_item_temp\"
            WHERE NOT EXISTS (
                SELECT 1 FROM \"Luv2_item\" WHERE \"Luv2_item\".\"code\" = \"Luv2_item_temp\".\"code\" and \"Luv2_item\".\"company_code\"=\"Luv2_item_temp\".\"company_code\") and \"Luv2_item_temp\".\"id_user\"='" . $userId . "' 
        ");

            #update item
            DB::statement("
            UPDATE \"Luv2_item\"
            SET \"barcode\"=\"Luv2_item_temp\".\"barcode\",\"name\"=\"Luv2_item_temp\".\"name\",\"brand_code\"=\"Luv2_item_temp\".\"brand_code\", \"category_code\"=\"Luv2_item_temp\".\"category_code\",\"size_code\"=\"Luv2_item_temp\".\"size_code\",\"status\"=\"Luv2_item_temp\".\"status\", \"inventory\"=\"Luv2_item_temp\".\"inventory\", \"sales\"=\"Luv2_item_temp\".\"sales\", \"purchase\"=\"Luv2_item_temp\".\"purchase\",\"active\"=\"Luv2_item_temp\".\"active\",\"avgprice\"=\"Luv2_item_temp\".\"avgprice\",\"price\"=\"Luv2_item_temp\".\"price\",\"consignment\"=\"Luv2_item_temp\".\"consignment\",\"supp_code\"=\"Luv2_item_temp\".\"supp_code\",\"updated_at\"=\"Luv2_item_temp\".\"updated_at\"
            FROM \"Luv2_item_temp\"
            WHERE \"Luv2_item_temp\".\"code\"=\"Luv2_item\".\"code\" AND \"Luv2_item_temp\".\"company_code\"=\"Luv2_item\".\"company_code\" AND \"Luv2_item_temp\".\"id_user\"='" . $userId . "'
            AND (
                \"Luv2_item_temp\".\"barcode\"<>\"Luv2_item\".\"barcode\" OR
                \"Luv2_item_temp\".\"name\"<>\"Luv2_item\".\"name\" OR
                \"Luv2_item_temp\".\"brand_code\"<>\"Luv2_item\".\"brand_code\" OR
                \"Luv2_item_temp\".\"category_code\"<>\"Luv2_item\".\"category_code\" OR
                \"Luv2_item_temp\".\"size_code\"<>\"Luv2_item\".\"size_code\" OR
                \"Luv2_item_temp\".\"status\"<>\"Luv2_item\".\"status\" OR
                \"Luv2_item_temp\".\"inventory\"<>\"Luv2_item\".\"inventory\" OR
                \"Luv2_item_temp\".\"sales\"<>\"Luv2_item\".\"sales\" OR
                \"Luv2_item_temp\".\"purchase\"<>\"Luv2_item\".\"purchase\" OR
                \"Luv2_item_temp\".\"active\"<>\"Luv2_item\".\"active\" OR
                \"Luv2_item_temp\".\"avgprice\"<>\"Luv2_item\".\"avgprice\" OR
                \"Luv2_item_temp\".\"price\"<>\"Luv2_item\".\"price\" OR
                \"Luv2_item_temp\".\"consignment\"<>\"Luv2_item\".\"consignment\" OR
                \"Luv2_item_temp\".\"supp_code\"<>\"Luv2_item\".\"supp_code\" OR
                \"Luv2_item_temp\".\"updated_at\"<>\"Luv2_item\".\"updated_at\"

            );
        ");

            $this->alert('success', 'Data item successfully saved', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            $this->dispatch('successImport');
        } catch (\Exception $e) {
            // DB::rollBack(); // Rollback the transaction on error

            // Error alert
            $this->alert('error', 'There was an error: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 5000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }


    public function updateChunk($index, $field, $value)
    {
        $this->chunks[$index][$field] = $value;
    }

    public function modeEditTrue()
    {
        $this->editChunks = true;
    }

    public function modeEditFalse()
    {
        $this->editChunks = false;
        $this->dispatch('refreshjs');
    }


    #[On('reGet')]
    public function mount()
    {
        $activeUDFs = DB::table('Luv2_udf')
            ->where('company_code', Auth::user()->company_code)
            ->where('active', 'Y')
            ->get();


        if ($activeUDFs->isNotEmpty()) {
            $this->activeUDFs = $activeUDFs;
            $this->udfFields = [];

            foreach ($activeUDFs as $udf) {
                $this->udfFields[$udf->field] = '';
            }
        } else {
            $this->activeUDFs = collect();
            $this->udfFields = [];
        }

        $this->itemSize = DB::table('Luv2_size')->where('company_code', Auth::user()->company_code)
            ->get();
        $this->itemBrand = DB::table('Luv2_brand')->where('company_code', Auth::user()->company_code)
            ->get();
        $this->itemCategory = DB::table('Luv2_category')->where('company_code', Auth::user()->company_code)
            ->get();
        $this->itemUnit = DB::table('Luv2_unit')->where('company_code', Auth::user()->company_code)
            ->get();
        $this->item = Item::where('company_code', Auth::user()->company_code)
            ->with([
                'brand' => function ($query) {
                    $query->where('company_code', Auth::user()->company_code);
                },
                'size' => function ($query) {
                    $query->where('company_code', Auth::user()->company_code);
                },
                'category' => function ($query) {
                    $query->where('company_code', Auth::user()->company_code);
                },
            ])
            ->get()
            ->map(function ($item) use ($activeUDFs) {
                if ($activeUDFs->isNotEmpty()) {
                    $itemUDFs = DB::table('Luv2_udf_item')
                        ->where('company_code', Auth::user()->company_code)
                        ->where('item_code', $item->code)
                        ->whereIn('code_udf', $activeUDFs->pluck('code'))
                        ->pluck('value', 'code_udf');

                    foreach ($activeUDFs as $udf) {
                        $item->{$udf->field} = $itemUDFs[$udf->code] ?? null;
                    }
                }

                return $item;
            });


        if ($this->item->isEmpty()) {
            $this->item = [];
        }

        if (Auth::user()->role != 'Admin') {
            $this->whs_code = Auth::user()->whs_code;
        } else {
            $this->whs_code = Warehouse::where('company_code', auth()->user()->company_code)->first()->code;
        }


        $tutorials = DB::table('Luv2_tutorial')
            ->where('id_user', auth()->user()->id)
            ->where('menu', 'Product')
            ->first();
        if ($tutorials && $tutorials->active === 'Y') {
            $this->istutorials = true;
        } else {
            $this->istutorials = false;
        }

        $this->suppliers = DB::table('Luv2_supplier')
            ->where('company_code', Auth::user()->company_code)
            ->where('consignment', 'Y')
            ->get();

        if(session()->has('fromCashier') && session('fromCashier') == auth()->id()){
            $this->dispatch('openModalBuat');     
        }
    }

    // public function updatedUnitCode()
    // {
    //     if ($this->unit_code == 'UnitCN') {
    //         $this->isCreateNewUnit = true;
    //     } else {
    //         $this->isCreateNewUnit = false;
    //         if ($this->unit_code != '') {
    //             $unit = DB::table('Luv2_unit')->where('id', $this->unit_code)->first();
    //             $this->unit_label = $unit->name;
    //         }
    //         $this->name_unit = '';
    //     }
    // }

    // public function updatedNameUnit()
    // {
    //     $this->unit_label = $this->name_unit;
    // }

    public function updatedSupplierCode()
    {
        if ($this->supplier_code == 'SupplierCN') {
            $this->isCreateNewSupplier = true;
        } else {
            $this->isCreateNewSupplier = false;
        }
    }

    public function updatedBrandCode()
    {
        if ($this->brand_code == 'BrandCN') {
            $this->isCreateNewBrand = true;
        } else {
            $this->isCreateNewBrand = false;
            $this->name_brand = '';
        }
    }

    public function updatedCategoryCode()
    {
        if ($this->category_code == 'CategoryCN') {
            $this->isCreateNewCategory = true;
        } else {
            $this->isCreateNewCategory = false;
            $this->name_category = '';
        }
    }

    public function updatedSizeCode()
    {
        if ($this->size_code == 'SizeCN') {
            $this->isCreateNewSize = true;
        } else {
            $this->isCreateNewSize = false;
            $this->name_size = '';
        }
    }
    public function save()
    {

        $this->validate();
        if ($this->isCreateNewBrand) {
            DB::table('Luv2_brand')->insert([
                'company_code' => Auth::user()->company_code,
                // 'whs_code' => $this->whs_code,
                'code' => $this->code_brand,
                'name' => $this->name_brand,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }

        if ($this->isCreateNewCategory) {
            DB::table('Luv2_category')->insert([
                'company_code' => Auth::user()->company_code,
                // 'whs_code' => $this->whs_code,
                'code' => $this->code_category,
                'name' => $this->name_category,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }

        if ($this->isCreateNewSize) {
            $lastFourDigits = substr(Auth::user()->company_code, -4);
            $tanggal = Carbon::now()->isoFormat('YYMMDDHHmm');
            DB::table('Luv2_size')->insert([
                'company_code' => Auth::user()->company_code,
                // 'whs_code' => $this->whs_code,
                'code' => $this->code_size,
                'name' => $this->name_size,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
                'unique_code' => $lastFourDigits . $this->code_size . $tanggal,
            ]);
        }
        // if ($this->isCreateNewUnit) {
        //     $units = DB::table('Luv2_unit')->where('company_code', auth()->user()->company_code)
        //         // ->where('whs_code', auth()->user()->whs_code)
        //         ->get();
        //     $count = $units->count();
        //     $id = ++$count;
        //     DB::table('Luv2_unit')->insert([
        //         'id' => $id,
        //         // 'whs_code' => $this->whs_code,
        //         'company_code' => Auth::user()->company_code,
        //         'name' => $this->name_unit,
        //         'created_at' => Carbon::now(),
        //         'updated_at' => Carbon::now(),
        //     ]);
        // }

        $submittedDataUDF = $this->udfFields;
        if ($submittedDataUDF) {
            foreach ($submittedDataUDF as $field => $value) {
                DB::table('Luv2_udf_item')->updateOrInsert(
                    [
                        'item_code' => $this->code,
                        'code_udf' => $this->activeUDFs->firstWhere('field', $field)->code ?? null,
                        'company_code' => Auth::user()->company_code,
                    ],
                    [
                        'value' => $value,
                        'updated_at' => Carbon::now(),
                        'created_at' => Carbon::now(),
                    ]
                );
            }
        }

        if ($this->isCreateNewSupplier) {
            $words = explode(' ', $this->suppliernew_name);
            $firstLetters = '';
            foreach ($words as $word) {
                $firstLetters .= substr($word, 0, 1);
            }
            $lastFourDigits = substr(Auth::user()->company_code, -4);
            $seconds = Carbon::now()->format('s');
            $this->supplier_code = $lastFourDigits . $firstLetters . $seconds;

            DB::table('Luv2_supplier')->insert([
                'company_code' => Auth::user()->company_code,
                'code' => $this->supplier_code,
                'name' => $this->suppliernew_name,
                'address' => $this->suppliernew_address,
                'phone' => $this->suppliernew_phone,
                'email' => $this->suppliernew_email,
                'consignment' => 'Y',
                'shared_margin' => $this->shared_margin,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now(),
            ]);
        }


        DB::table('Luv2_item')->insert([
            'company_code' => Auth::user()->company_code,
            // 'whs_code' => $this->whs_code,
            'code' => $this->code,
            'barcode' => $this->barcode,
            'name' => $this->name,
            'brand_code' => $this->code_brand !== '' ? $this->code_brand : $this->brand_code,
            'category_code' => $this->code_category !== '' ? $this->code_category : $this->category_code,
            'size_code' => $this->code_size !== '' ? $this->code_size : $this->size_code,
            'status' => 'Y',
            'inventory' => $this->inventory ? 'Y' : 'N',
            'sales' => $this->sales ? 'Y' : 'N',
            'purchase' => $this->purchase ? 'Y' : 'N',
            'consignment' => $this->isConsignment ? 'Y' : 'N',
            'supp_code' => $this->isConsignment ? $this->supplier_code : '',
            'active' => 'Y',
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now(),
        ]);
        // $unitcode = $this->unit_code == ''
        //     ? DB::table('Luv2_unit')->where('name', $this->name_unit)
        //         // ->where('whs_code', auth()->user()->whs_code)
        //         ->first()->id
        //     : $this->unit_code;

        // DB::table('Luv2_item_unit')->insert([
        //     'company_code' => Auth::user()->company_code,
        //     'whs_code' => $this->whs_code,
        //     'id_unit' => $unitcode,
        //     'item_code' => $this->code,
        //     'qty' => $this->qtyunit,
        //     'created_at' => Carbon::now(),
        //     'updated_at' => Carbon::now(),
        // ]);

        $this->dispatch('closeModal');
        $this->dispatch('afterSave');
        if(session()->has('toCashier') && session('toCashier') == auth()->id()) {
            session()->forget('toCashier');
            return redirect()->route('welcome.dashboard');
        }
    }

    #[On('closeModal')]
    public function closeModal()
    {
        $this->code = '';
        $this->barcode = '';
        $this->name = '';
        $this->brand_code = '';
        $this->category_code = '';
        $this->size_code = '';
        $this->isConsignment = false;
        $this->inventory = '';
        $this->sales = '';
        $this->purchase = '';
        $this->name_brand = '';
        $this->name_size = '';
        $this->name_category = '';
        $this->code_brand = '';
        $this->code_size = '';
        $this->code_category = '';
        $this->whs_code = '';
        $this->unit_code = '';
        $this->name_unit = '';
        $this->unit_label = '';
        $this->canEditSupplier = true;

        $this->dispatch('closemodalCreateItem');
        if ($this->isEdit) {
            $msg = 'Data item saved successfully';
        } else {
            $msg = 'Data item updated successfully';
        }

        $this->alert('success', $msg, [
            'position' => 'top-end',
            'timer' => 3000,
            'toast' => true,
            'timerProgressBar' => true,
        ]);

        $this->isEdit = false;

        $this->dispatch('reGet');
    }

    // public function closeModalEdit()
    // {
    //     $this->code = '';
    //     $this->barcode = '';
    //     $this->name = '';
    //     $this->brand_code = '';
    //     $this->category_code = '';
    //     $this->size_code = '';
    //     $this->inventory = '';
    //     $this->sales = '';
    //     $this->purchase = '';
    //     $this->name_brand = '';
    //     $this->name_size = '';
    //     $this->name_category = '';
    //     $this->code_brand = '';
    //     $this->code_size = '';
    //     $this->code_category = '';
    //     $this->whs_code = '';
    //     $this->unit_code = '';
    //     $this->name_unit = '';
    //     $this->unit_label = '';

    //     $this->dispatch('closemodalCreateItem');

    // }

    #[On('isEdit')]
    public function isEdit($code)
    {
        $item = DB::table('Luv2_item as a')
            ->select([
                'a.*',
                DB::raw('COALESCE((
                    SELECT SUM(qty)
                    FROM "Luv2_item_trans"
                    WHERE item_code = a.code
                    AND company_code = \'' . auth()->user()->company_code . '\'
                    AND whs_code = \'' . auth()->user()->whs_code . '\'
                ), 0) AS trans_qty')
            ])
            ->where('a.company_code', auth()->user()->company_code)
            ->where('a.code', $code)
            ->first();
        $isConsignment = $item->consignment == 'Y' ? true : false;

        if ($isConsignment) {
            $supplier = DB::table('Luv2_supplier')->where('code', $item->supp_code)->first();
            $this->supplier_code = $supplier->code;
        }

        $this->code = $item->code;
        $this->barcode = $item->barcode;
        $this->name = $item->name;
        $this->brand_code = $item->brand_code;
        $this->category_code = $item->category_code;
        $this->size_code = $item->size_code;
        $this->inventory = $item->inventory;
        $this->isConsignment = $isConsignment;
        $this->supplier_code = $isConsignment ? $supplier->code : '';
        $this->sales = $item->sales;
        $this->purchase = $item->purchase;

        // Fetch active UDFs for the item
        $activeUDFs = DB::table('Luv2_udf')
            ->where('company_code', Auth::user()->company_code)
            ->where('active', 'Y')
            ->get();

        if ($activeUDFs->isNotEmpty()) {
            foreach ($activeUDFs as $udf) {
                $udfValue = DB::table('Luv2_udf_item')
                    ->where('company_code', Auth::user()->company_code)
                    ->where('code_udf', $udf->code)
                    ->where('item_code', $item->code)
                    ->first();
                $this->udfFields[$udf->field] = $udfValue->value ?? '';
            }
        } else {
            $this->udfFields = [];
        }

        $this->isEdit = true;
        $this->canEditSupplier = ($item->trans_qty == 0) ? true : false;
        $this->dispatch('toOpenModal');
    }

    #[On('toOpenModal')]
    public function openModal()
    {
        $this->dispatch('openModalforEdit', [
            'code' => $this->code,
            'barcode' => $this->barcode,
            'name' => $this->name,
            'unit_code' => $this->unit_code,
            'qtyunit' => $this->qtyunit,
            'brand_code' => $this->brand_code,
            'category_code' => $this->category_code,
            'size_code' => $this->size_code,
            'inventory' => $this->inventory,
            'consignment' => $this->isConsignment,
            'supp_code' => $this->supplier_code,
            'sales' => $this->sales,
            'purchase' => $this->purchase,
        ]);
        $this->dispatch('refreshjs');
    }

    public function saveUpdate()
    {

        try {
            $this->validate($this->rules());

            if ($this->isCreateNewBrand) {
                DB::table('Luv2_brand')->insert([
                    'company_code' => Auth::user()->company_code,
                    'code' => $this->code_brand,
                    'name' => $this->name_brand,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ]);
            }

            if ($this->isCreateNewCategory) {
                DB::table('Luv2_category')->insert([
                    'company_code' => Auth::user()->company_code,
                    'code' => $this->code_category,
                    'name' => $this->name_category,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ]);
            }

            if ($this->isCreateNewSize) {
                $lastFourDigits = substr(Auth::user()->company_code, -4);
                $tanggal = Carbon::now()->isoFormat('YYMMDDHHmm');
                DB::table('Luv2_size')->insert([
                    'company_code' => Auth::user()->company_code,
                    'code' => $this->code_size,
                    'name' => $this->name_size,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                    'unique_code' => $lastFourDigits . $this->code_size . $tanggal,
                ]);
            }
            // if ($this->isCreateNewUnit) {
            //     $units = DB::table('Luv2_unit')->where('company_code', auth()->user()->company_code)
            //         // ->where('whs_code', auth()->user()->whs_code)
            //         ->get();
            //     $count = $units->count();
            //     $id = ++$count;
            //     DB::table('Luv2_unit')->insert([
            //         'id' => $id,
            //         // 'whs_code' => $this->whs_code,
            //         'company_code' => Auth::user()->company_code,
            //         'name' => $this->name_unit,
            //         'created_at' => Carbon::now(),
            //         'updated_at' => Carbon::now(),
            //     ]);
            //     $unitcode = $this->unit_code == ''
            //         ? DB::table('Luv2_unit')->where('name', $this->name_unit)
            //             // ->where('whs_code', auth()->user()->whs_code)
            //             ->first()->id
            //         : $this->unit_code;

            //     DB::table('Luv2_item_unit')->insert([
            //         'company_code' => Auth::user()->company_code,
            //         'whs_code' => $this->whs_code,
            //         'id_unit' => $unitcode,
            //         'item_code' => $this->code,
            //         'qty' => $this->qtyunit,
            //         'created_at' => Carbon::now(),
            //         'updated_at' => Carbon::now(),
            //     ]);
            // } else {
            //     // DB::table('Luv2_item_unit')
            //     //     ->where('item_code', $this->code)
            //     //     ->where('company_code', auth()->user()->company_code)
            //     //     ->where('whs_code', auth()->user()->whs_code)
            //     //     ->update([
            //     //         'id_unit' => $this->unit_code,
            //     //         'qty' => $this->qtyunit,
            //     //         'updated_at' => Carbon::now(),
            //     //     ]);

            //     DB::table('Luv2_item_unit')->updateOrInsert(
            //         [
            //             'item_code' => $this->code,
            //             'company_code' => auth()->user()->company_code,
            //             'whs_code' => auth()->user()->whs_code
            //         ],
            //         [
            //             'company_code' => Auth::user()->company_code,
            //             'whs_code' => auth()->user()->whs_code,
            //             'id_unit' => $this->unit_code,
            //             'qty' => $this->qtyunit,
            //             'updated_at' => Carbon::now(),
            //             'created_at' => Carbon::now(),
            //         ]
            //     );
            // }
            $submittedDataUDF = $this->udfFields;
            if ($submittedDataUDF) {
                foreach ($submittedDataUDF as $field => $value) {
                    DB::table('Luv2_udf_item')->updateOrInsert(
                        [
                            'item_code' => $this->code,
                            'code_udf' => $this->activeUDFs->firstWhere('field', $field)->code ?? null,
                            'company_code' => Auth::user()->company_code,
                        ],
                        [
                            'value' => $value,
                            'updated_at' => Carbon::now(),
                            'created_at' => Carbon::now(),
                        ]
                    );
                }
            }

            if ($this->isCreateNewSupplier) {
                $words = explode(' ', $this->suppliernew_name);
                $firstLetters = '';
                foreach ($words as $word) {
                    $firstLetters .= substr($word, 0, 1);
                }
                $lastFourDigits = substr(Auth::user()->company_code, -4);
                $seconds = Carbon::now()->format('s');
                $this->supplier_code = $lastFourDigits . $firstLetters . $seconds;

                DB::table('Luv2_supplier')->insert([
                    'company_code' => Auth::user()->company_code,
                    'code' => $this->supplier_code,
                    'name' => $this->suppliernew_name,
                    'address' => $this->suppliernew_address,
                    'phone' => $this->suppliernew_phone,
                    'email' => $this->suppliernew_email,
                    'consignment' => 'Y',
                    'shared_margin' => $this->shared_margin,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now(),
                ]);
            }

            $updateData = [
                'barcode' => $this->barcode,
                'name' => $this->name,
                'brand_code' => $this->code_brand !== '' ? $this->code_brand : $this->brand_code,
                'category_code' => $this->code_category !== '' ? $this->code_category : $this->category_code,
                'size_code' => $this->code_size !== '' ? $this->code_size : $this->size_code,
                'status' => 'Y',
                'inventory' => $this->inventory ? 'Y' : 'N',
                'sales' => $this->sales ? 'Y' : 'N',
                'purchase' => $this->purchase ? 'Y' : 'N',
                'active' => 'Y',
                'updated_at' => Carbon::now(),
            ];

            if ($this->canEditSupplier) {
                $updateData['consignment'] = $this->isConsignment ? 'Y' : 'N';
                $updateData['supp_code'] = $this->isConsignment ? $this->supplier_code : '';
            }

            DB::table('Luv2_item')
                ->where('company_code', auth()->user()->company_code)
                ->where('code', $this->code)
                ->update($updateData);


            $this->dispatch('closeModal');
            $this->dispatch('afterSave');
        } catch (\Exception $e) {
            dd($e);
        }
    }


    #[On('afterSave')]
    public function dataItemeafterSave()
    {
        $this->item = Item::where('company_code', Auth::user()->company_code)
            // ->where('whs_code', auth()->user()->whs_code)
            ->with(['brand', 'size', 'category'])
            ->get();

        $this->dispatch('refreshjs');
    }

    #[On('openSupplierModal')]
    public function openSupplierModal($code)
    {
        $item = DB::table('Luv2_item')->where('code', $code)->first();
        $supplier = DB::table('Luv2_supplier')->where('code', $item->supp_code)->first();
        $this->infoitem_name = $item->name;
        $this->infoitem_code = $item->code;
        $this->infoitem_barcode = $item->barcode;
        $this->infosupplier_name = $supplier->name;
        $this->infosupplier_code = $supplier->code;

        $this->dispatch('openModalInfoSupplier');
    }

    #[On('resetTheSession')]
    public function clearSes()
    {
        session()->forget('fromCashier');
    }
}
