<?php

namespace App\Livewire\Pos\Auth;

use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Livewire\Component;
use Jantinnerezo\LivewireAlert\LivewireAlert;

class SetupPassword extends Component
{
    use LivewireAlert;

    public $token;
    public $email;
    public $password;
    public $password_confirmation;

    public function mount($token)
    {
        $this->token = $token;
        
        // Get email from password_resets table
        $reset = DB::table('password_resets')
            ->where('token', $token)
            ->first();

        if (!$reset) {
            $this->alert('error', 'Invalid or expired token!', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return redirect()->to('/');
        }

        // Check if token is expired (24 hours)
        if (Carbon::parse($reset->created_at)->addHours(24)->isPast()) {
            $this->alert('error', 'Token has expired! Please request a new one.', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
            return redirect()->to('/');
        }

        $this->email = $reset->email;
    }

    public function setPassword()
    {
        $this->validate([
            'password' => 'required|min:6|confirmed',
            'password_confirmation' => 'required'
        ]);

        try {
            DB::beginTransaction();

            // Update password in supplier_dash
            DB::table('Luv2_supplier_dash')
                ->where('email', $this->email)
                ->update([
                    'password' => Hash::make($this->password),
                    'updated_at' => Carbon::now()
                ]);

            // Delete the used token
            DB::table('password_resets')
                ->where('token', $this->token)
                ->delete();

            DB::commit();

            $this->alert('success', 'Password set successfully! You can now login.', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);

            // Redirect to login page after 2 seconds
            $this->dispatch('redirect')->self();

        } catch (\Exception $e) {
            DB::rollBack();
            $this->alert('error', 'Failed to set password: ' . $e->getMessage(), [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'timerProgressBar' => true,
            ]);
        }
    }

    public function render()
    {
        return view('livewire.pos.auth.setup-password');
    }
} 