<?php

namespace App\Livewire\Payment;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\On;
use Livewire\Component;

class ModalRenew extends Component
{
    use LivewireAlert;

    public $company;
    public $price;
    public $selectedPackage = null; // Tracks the selected package.
    public $total = 0;
    public $totalWithPPN = 0;
    public $allUser;
    public $allRegularUser; // Non-admin users.
    public $paymentType = 'individual'; // Default payment type.
    public $selectedItem; // The currently selected package item.
    public $excludedUsers = []; 
    public $excludedUsersWithDetails = [];

    public function render()
    {
        return view('livewire.payment.modal-renew');
    }

    public function mount()
    {
        // Load company users and filter regular users (non-admins).
        $this->allUser = DB::table('Luv2_user')
            ->where('company_code', auth()->user()->company_code)
            ->where('active', 'Y')
            ->get();

        $this->allRegularUser = DB::table('Luv2_user')
            ->where('company_code', auth()->user()->company_code)
            ->where('code', '!=', auth()->user()->code)
            ->where('active', 'Y')
            ->get();

        // Load company and pricing data.
        $this->company = DB::table('Luv2_company')
            ->where('code', auth()->user()->company_code)
            ->first();

        $paket = DB::table('Luv2_paket')
            ->where('name', $this->company->plan)
            ->first();

        $this->price = DB::table('Luv2_paket_price')
            ->where('id_paket', $paket->id)
            ->get();
    }

    public function updatedSelectedPackage($value)
    {
        $userCount = $this->allUser->count() - count($this->excludedUsers);
        $selectedItem = collect($this->price)->firstWhere('id', $value);
        $this->selectedItem = $selectedItem;
        $this->total = $selectedItem ? $selectedItem->price * $userCount : 0;
    }

    public function updatedExcludedUsers()
    {
        $excludedUserDetails = DB::table('Luv2_user')
            ->whereIn('code', $this->excludedUsers)
            ->get(['code', 'username']); 

        $this->excludedUsersWithDetails = $excludedUserDetails->map(function ($user) {
            return [
                'code' => $user->code,
                'username' => $user->username, 
            ];
        })->toArray();
        $this->updateTotal();
    }

    public function updateTotal()
    {
        $userCount = $this->allUser->count() - count($this->excludedUsers);
        $this->total = $this->selectedItem ? $this->selectedItem->price * $userCount : 0;
    }

    public function openPpnModal()
    {
        $this->updateTotalWithPPN();
        $this->dispatch('show-ppn-modal');
    }

    public function backToPaymentModal()
    {
        $this->totalWithPPN = 0;
        $this->dispatch('show-payment-modal');
    }

    public function updatedPaymentType($value)
    {
        $this->updateTotalWithPPN();
    }

    public function updateTotalWithPPN()
    {
        if ($this->paymentType === 'business') {
            $this->totalWithPPN = $this->total + ($this->total * ($this->selectedItem->ppn ?? 0) / 100);
        } else {
            $this->totalWithPPN = $this->total;
        }
    }

    public function createDataPay()
    {
        \Midtrans\Config::$serverKey = config('services.midtrans.serverKey');
        \Midtrans\Config::$isProduction = config('services.midtrans.isProduction');
        \Midtrans\Config::$isSanitized = config('services.midtrans.isSanitized');
        \Midtrans\Config::$is3ds = config('services.midtrans.is3ds');

        $user = Auth::user();
        $companyCode = $user->company_code;

        $existingRenewal = DB::table('Luv2_renewal')
            ->where('company_code', $companyCode)
            ->where('status', 'pending')
            ->first();

        if ($existingRenewal) {
            $this->alert('warning', 'A pending payment exists. Please complete that payment first.', [
                'position' => 'top-end',
                'timer' => 3000,
                'toast' => true,
                'width' => '800',
                'timerProgressBar' => true,
            ]);
            return;
        }

        $company = DB::table('Luv2_company')->where('code', $companyCode)->first();
        if (!$company) {
            $this->alert('error', 'Company not found.', [
                'position' => 'top-center',
                'timer' => 3000,
                'toast' => true,
                'width' => '800',
                'timerProgressBar' => true,
            ]);
            return;
        }
        $orderId = $companyCode . '-' . \Carbon\Carbon::now()->format('YmdHis');
        $userAccountCount = $this->allUser->count() - count($this->excludedUsers);
        $accountLabel = $userAccountCount . ' Account' . ($userAccountCount > 1 ? 's' : '');
        $basePrice = $this->selectedItem ? $this->selectedItem->price : 0;
        $amount = $basePrice * $userAccountCount;

        $totalAmount = $this->paymentType === 'business'
            ? $amount + ($amount * $this->selectedItem->ppn / 100)
            : $amount;

        $params = [
            'transaction_details' => [
                'order_id' => $orderId,
                'gross_amount' => $totalAmount,
            ],
            'customer_details' => [
                'first_name' => $user->username,
                'last_name' => $company->name,
                'email' => $user->email,
                'phone' => $company->phone,
            ],
            'item_details' => [
                [
                    'id' => $orderId,
                    'price' => $totalAmount,
                    'quantity' => 1,
                    'name' => 'Subscription ' . $this->selectedItem->name . ' (' .  ucfirst($this->paymentType) . ') With '  . $accountLabel,
                ]
            ]
        ];

        try {
            $snapToken = \Midtrans\Snap::getSnapToken($params);
            $userCount = $this->allUser->count() - count($this->excludedUsers);

            DB::table('Luv2_renewal')->insert([
                'order_id' => $orderId,
                'snap_token' => $snapToken,
                'company_code' => $companyCode,
                'detail' => json_encode($params),
                'amount' => $totalAmount,
                'status' => 'pending',
                'user' => $userCount,
                'user_exclude' => json_encode($this->excludedUsersWithDetails),
                'plan' => $this->company->plan,
                'month' => $this->selectedItem->month,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $this->alert('success', 'Payment created successfully. Redirecting...');
            $this->dispatch('close-modals');
            $this->dispatch('redirecting');
        } catch (\Exception $e) {
            $this->alert('error', 'Failed to create payment. ' . $e->getMessage());
        }
    }

    #[On('redirecting')]
    public function afterDataPay()
    {
        sleep(1);
        return redirect()->route('paymentable');
    }
}
