<?php

namespace App\Livewire\Auth;

use Livewire\Component;
use Livewire\Attributes\Rule;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Carbon\Carbon;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Livewire\Attributes\On;

class Register extends Component
{

    #[Rule('required|string|max:255')]
    public $name;

    #[Rule('required|string|max:255|unique:Luv2_user,username')]
    public $username;

    #[Rule('required|string|max:255')]
    public $address;

    #[Rule('required|string|max:15')]
    public $phone;

    #[Rule('required|string|email|max:255|unique:Luv2_user,email')]
    public $email;

    #[Rule('required|string|min:8')]
    public $password;

    #[Rule('required|exists:Luv2_paket,name')]
    public $plan;

    public $code;

    public $terms;


    public function render()
    {
        return view('livewire.auth.register');
    }

    public function register()
    {
        $this->validate();

        $this->code = Carbon::now()->format('ymdHis');

        try {
            // Begin a transaction
            DB::beginTransaction();

            // Insert into Luv2_company table
            DB::table('Luv2_company')->insert([
                'code' => $this->code,
                'name' => $this->name,
                'address' => $this->address,
                'phone' => $this->phone,
                'email' => $this->email,
                'plan' => $this->plan,
                'updated_at' => now(),
            ]);

            $maxId = DB::table('Luv2_user')->max('id');


            // Insert into Luv2_user table
            DB::table('Luv2_user')->insert([
                'id' => $maxId + 1,
                'company_code' => $this->code,
                'whs_code' => '-',
                'code' => $this->code,
                'email' => $this->email,
                'username' => $this->username,
                'password' => Hash::make($this->password),
                'active' => 'Y',
                'role' => 'Admin',
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            $menus = [
                'Home',
                'Cashier',
                'Member',
                'Master',
                'Opname',
                'Goods Receipt',
                'Goods Return',
                'Item Transfer',
                'Dashboard',
                'Warehouse',
                'Supplier',
                'SalesEmployee',
                'Pricelist',
                'Product',
                'Promo',
                'Brand',
                'Tutorial',
            ];

            foreach ($menus as $menu) {
                DB::table('Luv2_tutorial')->insert([
                    'id_user' => $maxId + 1,
                    'menu' => $menu,
                    'active' => 'Y',
                ]);
            }

            // Commit the transaction
            DB::commit();

            // Dispatch the registration successful event
            session()->flash('message', 'Registration successful, please check your email to verify your account!');
            $this->dispatch('sendEmailVerification');
            // Flash success message to the session
            $this->dispatch('toHome');
        } catch (\Exception $e) {
            DB::rollBack();

            session()->flash('message', 'Failed to register new user!');
        }
    }

    #[On('sendEmailVerification')]
    public function sendEmailVerification()
    {
        try {
            $encryptedId = Crypt::encryptString($this->code);

            Mail::send('email.verify', ['name' => $this->username, 'companyname' => $this->name, 'companycode' => $encryptedId], function ($message) {
                $message->to($this->email)
                    ->subject('Verify Your Account and Company');
            });
        } catch (\Exception $e) {
            // dd($e);
        }
    }


    #[On('toHome')]
    public function toHome()
    {
        sleep(3);
        return redirect('/login');
    }
}
