<?php

namespace App\Console\Commands;

use App\Models\User;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Mail;
use Maatwebsite\Excel\Facades\Excel;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;
use App\Exports\ExportMergedInvoiceReport;
use Illuminate\Support\Facades\Log;

class SendInvoiceReport extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'report:send-invoice';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Send daily invoice report via email at 23:59';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $today = Carbon::yesterday();

        $users = User::whereNotNull('company_code')->where('role', 'Admin')->get();

        foreach ($users as $user) {
            // Get invoice data
            $query = DB::table('Luv2_invoice')
                ->leftJoin('Luv2_salesemp', 'Luv2_invoice.sales_code', '=', 'Luv2_salesemp.code')
                ->leftJoin('Luv2_user', 'Luv2_invoice.user_code', '=', 'Luv2_user.code')
                ->select(
                    'Luv2_invoice.*',
                    'Luv2_salesemp.name as sales_name',
                    'Luv2_user.username as users_name'
                )
                ->where('Luv2_invoice.company_code', $user->company_code)
                ->whereDate('Luv2_invoice.created_at', $today);

            $invoices = $query->get()->map(function ($item) {
                $item = (object)$item;
                
                if ($item->note) {
                    $invoiceNote = json_decode($item->note, true);
                    $item->cash = (float)($invoiceNote['cash'] ?? 0);
                    $item->transfer = (float)($invoiceNote['transfer'] ?? 0);
                } else {
                    $item->cash = 0;
                    $item->transfer = 0;
                }
                
                if ($item->cashback !== 'N') {
                    $cekvoucher = DB::table('Luv2_promo_voucher')->where('voucher', $item->cashback)->first();
                    $item->value = $cekvoucher ? (float)$cekvoucher->value : 0;
                } else {
                    $item->value = 0;
                }

                return $item;
            });

            if ($invoices->isEmpty()) {
                $this->info("No invoices found for {$user->email} ({$user->company_code})");
                continue;
            }

            $now = Carbon::now()->format('d-m-Y');
            
            // Prepare data for merged report
            $arData = [
                'startDate' => $today->format('Y-m-d'),
                'endDate' => $today->format('Y-m-d'),
                'type' => 'All',
                'search' => '',
                'whs_code' => $user->whs_code
            ];
            
            // Generate merged report with both sheets
            $mergedFilename = "Invoice_Report_{$user->company_code}_{$now}.xlsx";
            $mergedPath = "exports/{$mergedFilename}";
            
            Excel::store(new ExportMergedInvoiceReport($invoices, $arData), $mergedPath);

            // Send email with merged report
            Mail::send('emails.invoice-report-mail', ['user' => $user], function ($message) use ($user, $mergedPath, $mergedFilename) {
                $message->to([$user->email, 'adiganteng630@gmail.com', 'dama@softcomp.io'])
                    ->subject('Daily Invoice Report')
                    ->attach(Storage::path($mergedPath), [
                        'as' => $mergedFilename,
                        'mime' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'
                    ]);
            });

            Storage::delete($mergedPath);

            $this->info("Merged invoice report sent to {$user->email}");
            Log::info("Merged invoice report sent to {$user->email}");
        }
    }
}