'use client';

import { useState, useEffect } from 'react';
import { format, addMonths, startOfMonth, endOfMonth, eachDayOfInterval, isSameMonth, isToday } from 'date-fns';
import { ChevronLeft, ChevronRight } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { getSchedulesUrl, API_CONFIG } from '@/lib/config';

interface Schedule {
  date: string;
  time1: string;
  time2: string;
}

interface CalendarDay {
  date: Date;
  dayOfWeek: string;
  dayNumber: number;
  month: number;
  year: number;
  schedules: Schedule[];
}

export default function ScheduleCalendar() {
  const [currentMonth, setCurrentMonth] = useState(new Date());
  const [schedules, setSchedules] = useState<Schedule[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);

  // Fetch schedules from API
  useEffect(() => {
    const fetchSchedules = async () => {
      try {
        setLoading(true);
        const response = await fetch(getSchedulesUrl(), {
          headers: {
            'Authorization': `Bearer ${API_CONFIG.TOKEN}`,
            'Content-Type': 'application/json'
          }
        });
        
        if (!response.ok) {
          throw new Error('Failed to fetch schedules');
        }
        
        const data = await response.json();
        
        if (data.success) {
          setSchedules(data.data || []);
        } else {
          throw new Error(data.message || 'Failed to fetch schedules');
        }
      } catch (err) {
        console.error('Error fetching schedules:', err);
        setError(err instanceof Error ? err.message : 'An error occurred');
      } finally {
        setLoading(false);
      }
    };

    fetchSchedules();
  }, []);

  // Generate calendar data for current month and next month
  const generateCalendarDays = (month: Date): CalendarDay[] => {
    const start = startOfMonth(month);
    const end = endOfMonth(month);
    const days = eachDayOfInterval({ start, end });
    
    return days.map(date => {
      const dateStr = format(date, 'yyyy-MM-dd');
      const daySchedules = schedules.filter(schedule => schedule.date === dateStr);
      
      return {
        date,
        dayOfWeek: format(date, 'EEEE').toUpperCase(),
        dayNumber: date.getDate(),
        month: date.getMonth(),
        year: date.getFullYear(),
        schedules: daySchedules
      };
    });
  };

  const currentMonthDays = generateCalendarDays(currentMonth);
  const nextMonthDays = generateCalendarDays(addMonths(currentMonth, 1));

  const goToPreviousMonth = () => {
    setCurrentMonth(prev => addMonths(prev, -1));
  };

  const goToNextMonth = () => {
    setCurrentMonth(prev => addMonths(prev, 1));
  };

  const formatTime = (time: string) => {
    if (!time || time === '-') return '-';
    
    // Try to parse the time and format it
    try {
      const [hours, minutes] = time.split(':');
      const hour = parseInt(hours);
      const ampm = hour >= 12 ? 'PM' : 'AM';
      const displayHour = hour > 12 ? hour - 12 : hour === 0 ? 12 : hour;
      return `${displayHour}:${minutes} ${ampm}`;
    } catch {
      return time;
    }
  };

  if (loading) {
    return (
      <div className="flex justify-center items-center h-64">
        <div className="animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600"></div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="text-center py-8">
        <div className="text-red-600 mb-4">Error: {error}</div>
        <Button onClick={() => window.location.reload()}>Retry</Button>
      </div>
    );
  }

  return (
    <div className="space-y-8">
      {/* Month Navigation */}
      <div className="flex items-center justify-between">
        <Button
          variant="outline"
          onClick={goToPreviousMonth}
          className="flex items-center gap-2"
        >
          <ChevronLeft className="h-4 w-4" />
          Previous Month
        </Button>
        
        <div className="text-xl font-semibold text-gray-900">
          {format(currentMonth, 'MMMM yyyy')}
        </div>
        
        <Button
          variant="outline"
          onClick={goToNextMonth}
          className="flex items-center gap-2"
        >
          Next Month
          <ChevronRight className="h-4 w-4" />
        </Button>
      </div>

      {/* Calendar Grid */}
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        {/* Current Month */}
        <div className="bg-white rounded-lg shadow-md p-6">
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-lg font-semibold text-gray-900">
              {format(currentMonth, 'MMMM')}
            </h2>
            <div className="text-sm text-gray-500">
              {currentMonthDays.length} days
            </div>
          </div>
          
          <div className="grid grid-cols-7 gap-2">
            {/* Day headers */}
            {['SUN', 'MON', 'TUE', 'WED', 'THU', 'FRI', 'SAT'].map(day => (
              <div key={day} className="text-center text-xs font-medium text-gray-500 py-2">
                {day}
              </div>
            ))}
            
            {/* Calendar days */}
            {currentMonthDays.map((day, index) => (
              <div
                key={index}
                className={`min-h-[80px] p-2 rounded-lg border ${
                  isToday(day.date) 
                    ? 'border-blue-500 bg-blue-50' 
                    : 'border-gray-200 bg-gray-50'
                }`}
              >
                <div className="text-center">
                  <div className={`text-lg font-bold ${
                    isToday(day.date) ? 'text-blue-600' : 'text-gray-900'
                  }`}>
                    {day.dayNumber}
                  </div>
                  <div className="text-xs text-gray-600 bg-yellow-100 px-1 py-0.5 rounded mt-1">
                    {day.dayOfWeek}
                  </div>
                  
                  {/* Time slots */}
                  <div className="mt-2 space-y-1">
                    {day.schedules.length > 0 ? (
                      day.schedules.map((schedule, sIndex) => (
                        <div key={sIndex} className="text-xs">
                          <div className="bg-blue-100 text-blue-800 px-1 py-0.5 rounded">
                            {formatTime(schedule.time1)}
                          </div>
                          {schedule.time2 && schedule.time2 !== '-' && (
                            <div className="bg-blue-100 text-blue-800 px-1 py-0.5 rounded mt-1">
                              {formatTime(schedule.time2)}
                            </div>
                          )}
                        </div>
                      ))
                    ) : (
                      <div className="text-xs text-gray-400 bg-gray-100 px-1 py-0.5 rounded">
                        No slots
                      </div>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* Next Month */}
        <div className="bg-white rounded-lg shadow-md p-6">
          <div className="flex items-center justify-between mb-6">
            <h2 className="text-lg font-semibold text-gray-900">
              {format(addMonths(currentMonth, 1), 'MMMM')}
            </h2>
            <div className="text-sm text-gray-500">
              {nextMonthDays.length} days
            </div>
          </div>
          
          <div className="grid grid-cols-7 gap-2">
            {/* Day headers */}
            {['SUN', 'MON', 'TUE', 'WED', 'THU', 'FRI', 'SAT'].map(day => (
              <div key={day} className="text-center text-xs font-medium text-gray-500 py-2">
                {day}
              </div>
            ))}
            
            {/* Calendar days */}
            {nextMonthDays.map((day, index) => (
              <div
                key={index}
                className="min-h-[80px] p-2 rounded-lg border border-gray-200 bg-gray-50"
              >
                <div className="text-center">
                  <div className="text-lg font-bold text-gray-900">
                    {day.dayNumber}
                  </div>
                  <div className="text-xs text-gray-600 bg-yellow-100 px-1 py-0.5 rounded mt-1">
                    {day.dayOfWeek}
                  </div>
                  
                  {/* Time slots */}
                  <div className="mt-2 space-y-1">
                    {day.schedules.length > 0 ? (
                      day.schedules.map((schedule, sIndex) => (
                        <div key={sIndex} className="text-xs">
                          <div className="bg-blue-100 text-blue-800 px-1 py-0.5 rounded">
                            {formatTime(schedule.time1)}
                          </div>
                          {schedule.time2 && schedule.time2 !== '-' && (
                            <div className="bg-blue-100 text-blue-800 px-1 py-0.5 rounded mt-1">
                              {formatTime(schedule.time2)}
                            </div>
                          )}
                        </div>
                      ))
                    ) : (
                      <div className="text-xs text-gray-400 bg-gray-100 px-1 py-0.5 rounded">
                        No slots
                      </div>
                    )}
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>

      {/* Legend */}
      <div className="bg-white rounded-lg shadow-md p-6">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Schedule Legend</h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <div className="flex items-center gap-3">
            <div className="w-4 h-4 bg-blue-100 border border-blue-300 rounded"></div>
            <span className="text-sm text-gray-700">Available time slots</span>
          </div>
          <div className="flex items-center gap-3">
            <div className="w-4 h-4 bg-yellow-100 border border-yellow-300 rounded"></div>
            <span className="text-sm text-gray-700">Day of the week</span>
          </div>
          <div className="flex items-center gap-3">
            <div className="w-4 h-4 bg-blue-50 border border-blue-500 rounded"></div>
            <span className="text-sm text-gray-700">Today's date</span>
          </div>
        </div>
      </div>
    </div>
  );
} 