import React, { useState, useEffect } from 'react'
import { Input } from './ui/input'
import { Label } from './ui/label'
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from './ui/select'
import { getRecommendedPackagesForPerson } from '@/lib/masterDataHelpers'

interface ParticipantFormProps {
  selectedActivities: any
  reservationDays: any[]
  formData: any
  setFormData: (data: any) => void
  adultCount: number
  setAdultCount: (count: number) => void
  childrenCount: number
  setChildrenCount: (count: number) => void
  selectedPackages: any
  getDisplayPrice: (price: any) => string
  showAgentSection: boolean
  // Agent-specific props
  totalPax: number
  setTotalPax: (count: number) => void
  paxGroups: Array<{
    id: string;
    ageGroup: string;
    count: number;
    surfLevel: string;
    recommendedPackage: string;
  }>
  setPaxGroups: (groups: Array<{
    id: string;
    ageGroup: string;
    count: number;
    surfLevel: string;
    recommendedPackage: string;
  }>) => void
  addPaxGroup: () => void
  updatePaxGroup: (id: string, field: string, value: any) => void
  removePaxGroup: (id: string) => void
  getRemainingPax: () => number
  isPaxValid: () => boolean
  getPackageRequirementMessage: () => Array<{package: string, packageName: string, current: number, required: number, needed: number}> | null
}

export default function ParticipantForm({
  selectedActivities,
  reservationDays,
  formData,
  setFormData,
  adultCount,
  setAdultCount,
  childrenCount,
  setChildrenCount,
  selectedPackages,
  getDisplayPrice,
  showAgentSection,
  // Agent-specific props
  totalPax,
  setTotalPax,
  paxGroups,
  setPaxGroups,
  addPaxGroup,
  updatePaxGroup,
  removePaxGroup,
  getRemainingPax,
  isPaxValid,
  getPackageRequirementMessage
}: ParticipantFormProps) {

  // Console log untuk debugging
  // console.log('ParticipantForm - selectedPackages:', selectedPackages);
  return (
    <>
      {/* Header */}
      <div className="bg-gradient-to-r from-slate-50 to-blue-50 border-l-4 border-blue-400 rounded-lg p-3 sm:p-4 md:p-5 mb-4 sm:mb-5 md:mb-6 mt-8 sm:mt-10 md:mt-12 shadow-sm">
        <div className="flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-3 md:gap-4">
          <div className="w-8 h-8 sm:w-10 sm:h-10 md:w-12 md:h-12 bg-gradient-to-br from-blue-500 to-sky-500 rounded-lg flex items-center justify-center flex-shrink-0 self-center sm:self-auto">
            <svg className="w-4 h-4 sm:w-5 sm:h-5 md:w-6 md:h-6 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5H7a2 2 0 00-2 2v10a2 2 0 002 2h8a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2" />
            </svg>
          </div>
          <div className="text-center sm:text-left flex-1">
            <h2 className="text-base sm:text-lg md:text-xl lg:text-2xl font-semibold text-slate-800 leading-tight">Personal Information</h2>
            <p className="text-xs sm:text-sm md:text-base text-slate-600 mt-1 sm:mt-2 leading-relaxed">Please provide participant details and surf proficiency levels</p>
          </div>
        </div>
      </div>

      <div className="space-y-6 sm:space-y-8">
        {/* Participant Count Selectors */}
        {showAgentSection ? (
          /* Agent Mode - Number of Pax Interface */
          <div className="space-y-6">
            {/* Total Pax Input */}
            <div className="bg-gradient-to-r from-blue-50 to-indigo-50 border border-blue-200 rounded-lg p-4">
              <div className="flex flex-col sm:flex-row sm:items-center gap-4">
                <label className="text-sm font-medium text-gray-700 flex-shrink-0">
                  Total Number of Pax:
                </label>
                <input
                  type="number"
                  min="0"
                  value={totalPax.toString()}
                  onChange={(e) => {
                    const inputValue = e.target.value;
                    const count = parseInt(inputValue) || 0;
                    
                    // Calculate current total pax used by all groups
                    const currentUsedPax = paxGroups.reduce((sum, group) => sum + (group?.count || 0), 0);
                    
                    // Ensure total pax is not less than currently used pax
                    const validCount = Math.max(count, currentUsedPax);
                    
                    setTotalPax(validCount);
                    if (validCount === 0) {
                      setPaxGroups([]); // Clear pax groups when total pax changes to 0
                    }
                  }}
                  onKeyDown={(e) => {
                    // Prevent decreasing total pax below current usage
                    if (e.key === 'ArrowDown') {
                      const currentUsedPax = paxGroups.reduce((sum, group) => sum + (group?.count || 0), 0);
                      if (totalPax <= currentUsedPax) {
                        e.preventDefault();
                      }
                    }
                  }}
                  className={`w-24 px-3 py-2 text-sm border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors ${
                    paxGroups.reduce((sum, group) => sum + (group?.count || 0), 0) > totalPax
                      ? 'border-red-300 bg-red-50' 
                      : 'border-gray-300'
                  }`}
                  placeholder="0"
                />
                <div className="text-sm text-gray-600">
                  {totalPax > 0 && (
                    <span className="font-medium">
                      Remaining: {getRemainingPax()} pax
                    </span>
                  )}
                  {paxGroups.reduce((sum, group) => sum + (group?.count || 0), 0) > totalPax && (
                    <div className="mt-1 text-xs text-red-600">
                      ⚠️ Total pax cannot be less than allocated groups
                    </div>
                  )}
                </div>
              </div>
            </div>

            {/* Pax Groups Management */}
            {totalPax > 0 && (
              <div className="space-y-4">
                <div className="flex items-center justify-between">
                  <h3 className="text-lg font-semibold text-gray-800">Pax Groups</h3>
                </div>

                {/* Pax Groups List */}
                <div className="space-y-3">
                  {paxGroups.map((group, index) => {
                    if (!group) return null; // Safety check for undefined group
                    return (
                    <div key={group.id} className="bg-white border border-gray-200 rounded-lg p-4 shadow-sm">
                      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4">
                        {/* Age Group - Narrower */}
                        <div className="lg:col-span-1">
                          <label className="block text-xs font-medium text-gray-700 mb-1">
                            Age Group
                          </label>
                          <select
                            value={group.ageGroup}
                            onChange={(e) => updatePaxGroup(group.id, 'ageGroup', e.target.value)}
                            className="w-full px-2 py-2 text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                          >
                            <option value="adult">Adult</option>
                            <option value="teen">Teen (10-15)</option>
                            <option value="child">Child (6-10)</option>
                          </select>
                        </div>

                        {/* Number of Pax - Narrower */}
                        <div className="lg:col-span-1">
                          <label className="block text-xs font-medium text-gray-700 mb-1">
                            Pax
                          </label>
                          <input
                            type="number"
                            min="1"
                            max={getRemainingPax() + group.count}
                            value={group.count}
                            onChange={(e) => {
                              const inputValue = e.target.value;
                              const count = parseInt(inputValue) || 0;
                              
                              // Calculate current total pax used by other groups
                              const otherGroupsTotal = paxGroups
                                .filter(g => g && g.id !== group.id)
                                .reduce((sum, g) => sum + (g?.count || 0), 0);
                              
                              // Calculate maximum allowed for this group
                              const maxAllowed = Math.max(1, totalPax - otherGroupsTotal);
                              
                              // Ensure count is within valid range
                              const validCount = Math.max(1, Math.min(count, maxAllowed));
                              
                              updatePaxGroup(group.id, 'count', validCount);
                            }}
                            onKeyDown={(e) => {
                              // Prevent typing values that would exceed total pax
                              if (e.key === 'ArrowUp' || e.key === 'ArrowDown') {
                                const currentCount = group.count;
                                const otherGroupsTotal = paxGroups
                                  .filter(g => g && g.id !== group.id)
                                  .reduce((sum, g) => sum + (g?.count || 0), 0);
                                const maxAllowed = Math.max(1, totalPax - otherGroupsTotal);
                                
                                if (e.key === 'ArrowUp' && currentCount >= maxAllowed) {
                                  e.preventDefault();
                                }
                                if (e.key === 'ArrowDown' && currentCount <= 1) {
                                  e.preventDefault();
                                }
                              }
                            }}
                            className={`w-full px-2 py-2 text-sm border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 ${
                              getRemainingPax() === 0 && group.count > 0 
                                ? 'border-yellow-300 bg-yellow-50' 
                                : 'border-gray-300'
                            }`}
                          />
                          {/* Show warning when approaching limit */}
                          {getRemainingPax() === 0 && group.count > 0 && (
                            <div className="mt-1 text-xs text-yellow-600">
                              ⚠️ All pax allocated
                            </div>
                          )}
                        </div>

                        {/* Surf Level - Narrower */}
                        <div className="lg:col-span-1">
                          <label className="block text-xs font-medium text-gray-700 mb-1">
                            Level
                          </label>
                          <select
                            value={group.surfLevel}
                            onChange={(e) => updatePaxGroup(group.id, 'surfLevel', e.target.value)}
                            className="w-full px-2 py-2 text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                          >
                            <option value="beginner">Beginner</option>
                            <option value="intermediate">Intermediate</option>
                            <option value="advanced">Advanced</option>
                          </select>
                        </div>

                        {/* Recommended Package - Wider */}
                        <div className="lg:col-span-2">
                          <label className="block text-xs font-medium text-gray-700 mb-1">
                            Package
                          </label>
                          <div className="flex gap-2">
                            <select
                              value={group.recommendedPackage}
                              onChange={(e) => updatePaxGroup(group.id, 'recommendedPackage', e.target.value)}
                              className="flex-1 px-3 py-2 text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                            >
                              <option value="">Select Package</option>
                              {(() => {
                                let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
                                if (selectedActivities.surfLessons && selectedActivities.surfTours) {
                                  typeCriteria = ['L', 'ST'];
                                } else if (selectedActivities.surfTours) {
                                  typeCriteria = 'ST';
                                } else {
                                  typeCriteria = 'L';
                                }

                                const groupCriteria = {
                                  level: (group.surfLevel === 'beginner' ? 'B' : group.surfLevel === 'intermediate' ? 'I' : 'A'),
                                  age: (group.ageGroup === 'adult' ? 'Ad' : group.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
                                  type: typeCriteria,
                                  bookingDate: reservationDays[0]?.date,
                                  personCount: group.count,
                                };

                                const recommendedPackages = getRecommendedPackagesForPerson(groupCriteria);
                                const uniquePackages = recommendedPackages.reduce((acc: any[], pkg: any) => {
                                  const existingPackage = acc.find(existing => existing.code === pkg.code);
                                  if (!existingPackage) {
                                    acc.push(pkg);
                                  }
                                  return acc;
                                }, []);

                                return uniquePackages.map((pkg: any) => (
                                  <option key={pkg.code} value={pkg.code}>
                                    {pkg.name} - {getDisplayPrice(pkg.price)}
                                  </option>
                                ));
                              })()}
                            </select>
                            {/* Remove Button */}
                            <button
                              type="button"
                              onClick={() => removePaxGroup(group.id)}
                              className="px-3 py-2 text-sm text-red-600 hover:text-red-800 hover:bg-red-50 rounded transition-colors flex items-center gap-1 flex-shrink-0"
                            >
                              <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                              </svg>
                              <span className="hidden sm:inline">Remove</span>
                            </button>
                          </div>
                        </div>
                      </div>

                      {/* Package Selection Help Text */}
                      {!group.recommendedPackage && (
                        <div className="mt-1 text-xs text-gray-500">
                          💡 You can select any package even if this group doesn't have enough people. Missing people can be added to other groups.
                        </div>
                      )}

                      {/* Package Min Pax Info - Below Package */}
                      {group.recommendedPackage && (() => {
                        let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
                        if (selectedActivities.surfLessons && selectedActivities.surfTours) {
                          typeCriteria = ['L', 'ST'];
                        } else if (selectedActivities.surfTours) {
                          typeCriteria = 'ST';
                        } else {
                          typeCriteria = 'L';
                        }

                        const groupCriteria = {
                          level: (group.surfLevel === 'beginner' ? 'B' : group.surfLevel === 'intermediate' ? 'I' : 'A'),
                          age: (group.ageGroup === 'adult' ? 'Ad' : group.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
                          type: typeCriteria,
                          bookingDate: reservationDays[0]?.date,
                          personCount: group.count,
                        };

                        const recommendedPackages = getRecommendedPackagesForPerson(groupCriteria);
                        const selectedPackage = recommendedPackages.find((pkg: any) => pkg.code === group.recommendedPackage);

                        if (selectedPackage && selectedPackage.pax) {
                          const requiredPax = selectedPackage.pax;

                          // Count total people across ALL groups who selected this package
                          const totalPeopleWithSamePackage = paxGroups
                            .filter(g => g && g.recommendedPackage === group.recommendedPackage)
                            .reduce((sum, g) => sum + (g?.count || 0), 0);

                          if (totalPeopleWithSamePackage < requiredPax || totalPeopleWithSamePackage % requiredPax !== 0) {
                            let needed = 0;
                            let message = "";

                            if (totalPeopleWithSamePackage < requiredPax) {
                              needed = requiredPax - totalPeopleWithSamePackage;
                              message = `Add ${needed} more people across all groups to meet the minimum requirement.`;
                            } else {
                              needed = requiredPax - (totalPeopleWithSamePackage % requiredPax);
                              message = `Add ${needed} more people across all groups to complete the group (must be multiples of ${requiredPax}).`;
                            }

                            return (
                              <div className="mt-2 p-2 bg-yellow-50 border border-yellow-200 rounded text-xs text-yellow-800">
                                <div className="flex items-center gap-1">
                                  <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                                  </svg>
                                  <div>
                                    <div className="font-medium">
                                      This package requires {requiredPax} people or multiples. Currently {totalPeopleWithSamePackage} people across all groups.
                                    </div>
                                    <div className="mt-1">
                                      {message}
                                    </div>
                                    <div className="mt-1 text-yellow-700">
                                      💡 You can add more people to any group to meet the requirement.
                                    </div>
                                  </div>
                                </div>
                              </div>
                            );
                          }
                        }
                        return null;
                      })()}
                    </div>
                    );
                  })}
                </div>

                {/* Global Package Requirement Messages */}
                {getPackageRequirementMessage() && getPackageRequirementMessage()!.map((req, index) => (
                  <div key={index} className="bg-red-50 border border-red-200 rounded-lg p-3">
                    <div className="flex items-center gap-2">
                      <svg className="w-5 h-5 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                      </svg>
                      <div className="text-sm text-red-800">
                        <div className="font-medium">
                          Package "{req.packageName}" needs {req.needed} more people across all groups.
                        </div>
                        <div className="mt-1 text-xs text-red-600">
                          💡 Add people to any group to meet the requirement.
                        </div>
                      </div>
                    </div>
                  </div>
                ))}

                {/* Validation Messages (Total Pax Mismatch) */}
                {totalPax > 0 && !isPaxValid() && (
                  <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-3">
                    <div className="flex items-center gap-2">
                      <svg className="w-5 h-5 text-yellow-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                      </svg>
                      <span className="text-sm text-yellow-800">
                        Total pax in groups ({paxGroups.reduce((sum, group) => sum + (group?.count || 0), 0)}) must equal total pax ({totalPax})
                      </span>
                    </div>
                  </div>
                )}

                {/* Add Group Button - Now placed below existing groups */}
                <div className="flex justify-center pt-2">
                  <button
                    type="button"
                    onClick={addPaxGroup}
                    disabled={getRemainingPax() <= 0}
                    className="px-6 py-3 bg-blue-600 text-white text-sm font-medium rounded-lg hover:bg-blue-700 disabled:bg-gray-300 disabled:cursor-not-allowed transition-colors flex items-center gap-2 shadow-md hover:shadow-lg"
                  >
                    <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 4v16m8-8H4" />
                    </svg>
                    Add Group
                  </button>
                </div>
              </div>
            )}
          </div>
        ) : (
          /* Customer Mode - Original Interface */
          <div className="space-y-4 sm:space-y-6">
            <div className="flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-4">
              <label className="text-xs sm:text-sm font-medium text-gray-700">Number of Adults:</label>
              <select
                value={adultCount.toString()}
                onChange={(e) => {
                  const count = parseInt(e.target.value);
                  setAdultCount(count);
                  setFormData((prev: any) => ({
                    ...prev,
                    adults: Array.from({ length: count }, (_, i) => ({
                      name: prev.adults[i]?.name || "",
                      email: prev.adults[i]?.email || "",
                      phone: prev.adults[i]?.phone || "",
                      surfLevel: prev.adults[i]?.surfLevel || "beginner",
                      recommendation: prev.adults[i]?.recommendation || "",
                      medical: prev.adults[i]?.medical || ["no_medical"],
                      medical_other: prev.adults[i]?.medical_other || ""
                    }))
                  }));
                }}
                className="w-20 px-2 sm:px-3 py-2 text-xs sm:text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
              >
                <option value="0">0</option>
                {Array.from({ length: 10 }, (_, i) => (
                  <option key={i + 1} value={(i + 1).toString()}>{i + 1}</option>
                ))}
              </select>
            </div>
            
            <div className="flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-4">
              <label className="text-xs sm:text-sm font-medium text-gray-700">Number of Children:</label>
              <select
                value={childrenCount.toString()}
                onChange={(e) => {
                  const count = parseInt(e.target.value);
                  setChildrenCount(count);
                  setFormData((prev: any) => ({
                    ...prev,
                    children: Array.from({ length: count }, (_, i) => ({
                      name: prev.children[i]?.name || "",
                      email: prev.children[i]?.email || "",
                      phone: prev.children[i]?.phone || "",
                      ageGroup: prev.children[i]?.ageGroup || "child",
                      surfLevel: prev.children[i]?.surfLevel || "beginner",
                      recommendation: prev.children[i]?.recommendation || "",
                      medical: prev.children[i]?.medical || ["no_medical"],
                      medical_other: prev.children[i]?.medical_other || ""
                    }))
                  }));
                }}
                className="w-20 px-2 sm:px-3 py-2 text-xs sm:text-sm border border-gray-300 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
              >
                <option value="0">0</option>
                {Array.from({ length: 10 }, (_, i) => (
                  <option key={i + 1} value={(i + 1).toString()}>{i + 1}</option>
                ))}
              </select>
            </div>
          </div>
        )}

        {/* All Participants - Adults First, Then Children - Hidden for Agent Booking */}
        {!showAgentSection && (
          <div className="space-y-4 sm:space-y-6">
          {/* Adult Participants */}
                  {(formData.adults || []).slice(0, adultCount || 0).map((adult: any, idx: number) => {
            return (
              <div key={`adult-${idx}`} className="bg-white border border-gray-200 rounded-lg p-4 sm:p-6 shadow-sm">
                <div className="flex items-center gap-2 sm:gap-3 mb-3 sm:mb-4">
                  <div className="w-6 h-6 sm:w-8 sm:h-8 bg-blue-500 text-white rounded-full flex items-center justify-center text-xs sm:text-sm font-medium">
                    {idx + 1}
                  </div>
                  <h4 className="text-sm sm:text-base font-medium text-gray-900">Adult {idx + 1}</h4>
                </div>

                <div className="space-y-4 sm:space-y-6">
                  {/* Form Section */}
                  <div>
                    <div className="grid grid-cols-1 sm:grid-cols-12 gap-4 sm:gap-6">
                      {/* Name */}
                      <div className="sm:col-span-4 space-y-2">
                        <Label htmlFor={`adult-name-${idx}`} className="text-xs sm:text-sm font-medium text-gray-700">
                          Full Name *
                        </Label>
                        <Input
                          id={`adult-name-${idx}`}
                          type="text"
                          value={adult.name || ""}
                          onChange={(e) => {
                            const newAdults = [...formData.adults];
                            newAdults[idx] = { ...newAdults[idx], name: e.target.value };
                            setFormData({ ...formData, adults: newAdults });
                          }}
                          placeholder="Enter full name"
                          className="h-8 sm:h-10 text-xs sm:text-sm"
                        />
                      </div>

                      {/* Surf Level */}
                      <div className="sm:col-span-3 space-y-2">
                        <Label htmlFor={`adult-surf-level-${idx}`} className="text-xs sm:text-sm font-medium text-gray-700">
                          Surf Level *
                        </Label>
                        <Select
                          value={adult.surfLevel || "beginner"}
                          onValueChange={(value) => {
                            const newAdults = [...formData.adults];
                            newAdults[idx] = { ...newAdults[idx], surfLevel: value };
                            setFormData({ ...formData, adults: newAdults });
                          }}
                        >
                          <SelectTrigger className="h-8 sm:h-10 text-xs sm:text-sm">
                            <SelectValue placeholder="Select surf level" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="beginner">Beginner</SelectItem>
                            <SelectItem value="intermediate">Intermediate</SelectItem>
                            <SelectItem value="advanced">Advanced</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>

                      {/* Package Selection */}
                      <div className="sm:col-span-5 space-y-2">
                        <Label htmlFor={`adult-package-${idx}`} className="text-xs sm:text-sm font-medium text-gray-700">
                          Package *
                        </Label>
                        <Select
                          value={adult.recommendation || ""}
                          onValueChange={(value) => {
                            const newAdults = [...formData.adults];
                            newAdults[idx] = { ...newAdults[idx], recommendation: value };
                            setFormData({ ...formData, adults: newAdults });
                          }}
                        >
                          <SelectTrigger className="h-8 sm:h-10 text-xs sm:text-sm">
                            <SelectValue placeholder="Select package" />
                          </SelectTrigger>
                          <SelectContent>
                            {(() => {
                              let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
                              if (selectedActivities.surfLessons && selectedActivities.surfTours) {
                                typeCriteria = ['L', 'ST'];
                              } else if (selectedActivities.surfTours) {
                                typeCriteria = 'ST';
                              } else {
                                typeCriteria = 'L';
                              }

                              const personCriteria = {
                                level: (adult.surfLevel === 'beginner' ? 'B' : adult.surfLevel === 'intermediate' ? 'I' : 'A'),
                                age: 'Ad' as 'A' | 'Ad' | 'Ch1' | 'Ch2',
                                type: typeCriteria,
                                bookingDate: reservationDays[0]?.date,
                                personCount: (adultCount || 0) + (childrenCount || 0), // Total people for minimum pax validation
                              };

                              const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
                              const uniquePackages = recommendedPackages.reduce((acc: any[], pkg: any) => {
                                const existingPackage = acc.find(existing => existing.code === pkg.code);
                                if (!existingPackage) {
                                  acc.push(pkg);
                                }
                                return acc;
                              }, []);

                              return uniquePackages.map((pkg: any) => (
                                <SelectItem key={pkg.code} value={pkg.code}>
                                  {pkg.name} - {getDisplayPrice(pkg.price)}
                                </SelectItem>
                              ));
                            })()}
                          </SelectContent>
                        </Select>
                      </div>
                    </div>
                    
                    {/* Package Requirements Warning - Full Width */}
                    {(() => {
                      if (!adult.recommendation) return null;
                      
                      // Find package details
                      let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
                      if (selectedActivities.surfLessons && selectedActivities.surfTours) {
                        typeCriteria = ['L', 'ST'];
                      } else if (selectedActivities.surfTours) {
                        typeCriteria = 'ST';
                      } else {
                        typeCriteria = 'L';
                      }

                      const personCriteria = {
                        level: (adult.surfLevel === 'beginner' ? 'B' : adult.surfLevel === 'intermediate' ? 'I' : 'A'),
                        age: 'Ad' as 'A' | 'Ad' | 'Ch1' | 'Ch2',
                        type: typeCriteria,
                        bookingDate: reservationDays[0]?.date,
                        personCount: (adultCount || 0) + (childrenCount || 0),
                      };

                      const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
                      const selectedPackage = recommendedPackages.find((pkg: any) => pkg.code === adult.recommendation);
                      
                      if (selectedPackage && selectedPackage.pax) {
                        const requiredPax = selectedPackage.pax;
                        const totalPeopleWithSamePackage = [
                          ...(formData.adults || []).filter((a: any) => a && a.recommendation === adult.recommendation),
                          ...(formData.children || []).filter((c: any) => c && c.recommendation === adult.recommendation)
                        ].length;

                        if (totalPeopleWithSamePackage < requiredPax || totalPeopleWithSamePackage % requiredPax !== 0) {
                          let needed = 0;
                          if (totalPeopleWithSamePackage < requiredPax) {
                            needed = requiredPax - totalPeopleWithSamePackage;
                          } else {
                            needed = requiredPax - (totalPeopleWithSamePackage % requiredPax);
                          }

                          return (
                            <div className="mt-3 p-3 bg-yellow-50 border border-yellow-200 rounded-lg text-xs text-yellow-800">
                              <div className="flex items-center gap-2">
                                <svg className="w-4 h-4 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                                </svg>
                                <div>
                                  <div className="font-medium">
                                    Package Requirements Not Met: "{selectedPackage.name}" requires {requiredPax} people. Currently {totalPeopleWithSamePackage} people selected.
                                  </div>
                                  <div className="mt-1">
                                    Add {needed} more people to meet the minimum requirement.
                                  </div>
                                </div>
                              </div>
                            </div>
                          );
                        }
                      }
                      return null;
                    })()}
                  </div>

                  {/* Medical Conditions Section - Always show */}
                  <MedicalConditionsAccordion
                    participant={adult}
                    participantIndex={idx}
                    isChild={false}
                    formData={formData}
                    setFormData={setFormData}
                  />
                </div>
              </div>
            );
          })}

          {/* Child Participants */}
                  {(formData.children || []).slice(0, childrenCount || 0).map((child: any, idx: number) => {
            return (
              <div key={`child-${idx}`} className="bg-white border border-gray-200 rounded-lg p-4 sm:p-6 shadow-sm">
                <div className="flex items-center gap-2 sm:gap-3 mb-3 sm:mb-4">
                  <div className="w-6 h-6 sm:w-8 sm:h-8 bg-green-500 text-white rounded-full flex items-center justify-center text-xs sm:text-sm font-medium">
                    {idx + 1}
                  </div>
                  <h4 className="text-sm sm:text-base font-medium text-gray-900">Child {idx + 1}</h4>
                </div>

                <div className="space-y-4 sm:space-y-6">
                  {/* Form Section */}
                  <div>
                    <div className="grid grid-cols-1 sm:grid-cols-12 gap-4 sm:gap-6">
                      {/* Name */}
                      <div className="sm:col-span-3 space-y-2">
                        <Label htmlFor={`child-name-${idx}`} className="text-xs sm:text-sm font-medium text-gray-700">
                          Full Name *
                        </Label>
                        <Input
                          id={`child-name-${idx}`}
                          type="text"
                          value={child.name || ""}
                          onChange={(e) => {
                            const newChildren = [...formData.children];
                            newChildren[idx] = { ...newChildren[idx], name: e.target.value };
                            setFormData({ ...formData, children: newChildren });
                          }}
                          placeholder="Enter full name"
                          className="h-8 sm:h-10 text-xs sm:text-sm"
                        />
                      </div>

                      {/* Age Group */}
                      <div className="sm:col-span-2 space-y-2">
                        <Label htmlFor={`child-age-group-${idx}`} className="text-xs sm:text-sm font-medium text-gray-700">
                          Age *
                        </Label>
                        <Select
                          value={child.ageGroup || "child"}
                          onValueChange={(value) => {
                            const newChildren = [...formData.children];
                            newChildren[idx] = { ...newChildren[idx], ageGroup: value };
                            setFormData({ ...formData, children: newChildren });
                          }}
                        >
                          <SelectTrigger className="h-8 sm:h-10 text-xs sm:text-sm">
                            <SelectValue placeholder="Age" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="child">6-10</SelectItem>
                            <SelectItem value="teen">11-15</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>

                      {/* Surf Level */}
                      <div className="sm:col-span-2 space-y-2">
                        <Label htmlFor={`child-surf-level-${idx}`} className="text-xs sm:text-sm font-medium text-gray-700">
                          Level *
                        </Label>
                        <Select
                          value={child.surfLevel || "beginner"}
                          onValueChange={(value) => {
                            const newChildren = [...formData.children];
                            newChildren[idx] = { ...newChildren[idx], surfLevel: value };
                            setFormData({ ...formData, children: newChildren });
                          }}
                        >
                          <SelectTrigger className="h-8 sm:h-10 text-xs sm:text-sm">
                            <SelectValue placeholder="Level" />
                          </SelectTrigger>
                          <SelectContent>
                            <SelectItem value="beginner">Beginner</SelectItem>
                            <SelectItem value="intermediate">Intermediate</SelectItem>
                            <SelectItem value="advanced">Advanced</SelectItem>
                          </SelectContent>
                        </Select>
                      </div>

                      {/* Package Selection */}
                      <div className="sm:col-span-5 space-y-2">
                        <Label htmlFor={`child-package-${idx}`} className="text-xs sm:text-sm font-medium text-gray-700">
                          Package *
                        </Label>
                        <Select
                          value={child.recommendation || ""}
                          onValueChange={(value) => {
                            const newChildren = [...formData.children];
                            newChildren[idx] = { ...newChildren[idx], recommendation: value };
                            setFormData({ ...formData, children: newChildren });
                          }}
                        >
                          <SelectTrigger className="h-8 sm:h-10 text-xs sm:text-sm">
                            <SelectValue placeholder="Select package" />
                          </SelectTrigger>
                          <SelectContent>
                            {(() => {
                              let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
                              if (selectedActivities.surfLessons && selectedActivities.surfTours) {
                                typeCriteria = ['L', 'ST'];
                              } else if (selectedActivities.surfTours) {
                                typeCriteria = 'ST';
                              } else {
                                typeCriteria = 'L';
                              }

                              const personCriteria = {
                                level: (child.surfLevel === 'beginner' ? 'B' : child.surfLevel === 'intermediate' ? 'I' : 'A'),
                                age: (child.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
                                type: typeCriteria,
                                bookingDate: reservationDays[0]?.date,
                                personCount: (adultCount || 0) + (childrenCount || 0), // Total people for minimum pax validation
                              };

                              const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
                              const uniquePackages = recommendedPackages.reduce((acc: any[], pkg: any) => {
                                const existingPackage = acc.find(existing => existing.code === pkg.code);
                                if (!existingPackage) {
                                  acc.push(pkg);
                                }
                                return acc;
                              }, []);

                              return uniquePackages.map((pkg: any) => (
                                <SelectItem key={pkg.code} value={pkg.code}>
                                  {pkg.name} - {getDisplayPrice(pkg.price)}
                                </SelectItem>
                              ));
                            })()}
                          </SelectContent>
                        </Select>
                      </div>
                    </div>
                    
                    {/* Package Requirements Warning - Full Width */}
                    {(() => {
                      if (!child.recommendation) return null;
                      
                      // Find package details
                      let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
                      if (selectedActivities.surfLessons && selectedActivities.surfTours) {
                        typeCriteria = ['L', 'ST'];
                      } else if (selectedActivities.surfTours) {
                        typeCriteria = 'ST';
                      } else {
                        typeCriteria = 'L';
                      }

                      const personCriteria = {
                        level: (child.surfLevel === 'beginner' ? 'B' : child.surfLevel === 'intermediate' ? 'I' : 'A'),
                        age: (child.ageGroup === 'teen' ? 'Ch2' : 'Ch1') as 'A' | 'Ad' | 'Ch1' | 'Ch2',
                        type: typeCriteria,
                        bookingDate: reservationDays[0]?.date,
                        personCount: (adultCount || 0) + (childrenCount || 0),
                      };

                      const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
                      const selectedPackage = recommendedPackages.find((pkg: any) => pkg.code === child.recommendation);
                      
                      if (selectedPackage && selectedPackage.pax) {
                        const requiredPax = selectedPackage.pax;
                        const totalPeopleWithSamePackage = [
                          ...(formData.adults || []).filter((a: any) => a && a.recommendation === child.recommendation),
                          ...(formData.children || []).filter((c: any) => c && c.recommendation === child.recommendation)
                        ].length;

                        if (totalPeopleWithSamePackage < requiredPax || totalPeopleWithSamePackage % requiredPax !== 0) {
                          let needed = 0;
                          if (totalPeopleWithSamePackage < requiredPax) {
                            needed = requiredPax - totalPeopleWithSamePackage;
                          } else {
                            needed = requiredPax - (totalPeopleWithSamePackage % requiredPax);
                          }

                          return (
                            <div className="mt-3 p-3 bg-yellow-50 border border-yellow-200 rounded-lg text-xs text-yellow-800">
                              <div className="flex items-center gap-2">
                                <svg className="w-4 h-4 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                                </svg>
                                <div>
                                  <div className="font-medium">
                                    Package Requirements Not Met: "{selectedPackage.name}" requires {requiredPax} people. Currently {totalPeopleWithSamePackage} people selected.
                                  </div>
                                  <div className="mt-1">
                                    Add {needed} more people to meet the minimum requirement.
                                  </div>
                                </div>
                              </div>
                            </div>
                          );
                        }
                      }
                      return null;
                    })()}
                  </div>

                  {/* Medical Conditions Section - Always show */}
                  <MedicalConditionsAccordion
                    participant={child}
                    participantIndex={idx}
                    isChild={true}
                    formData={formData}
                    setFormData={setFormData}
                  />
                </div>
              </div>
            );
          })}
          </div>
        )}

        {/* Package Validation for Customer Mode */}
        {!showAgentSection && (
          <div className="space-y-4">
            {(() => {
              // Get all selected packages from participants
              const selectedPackageCodes = [
                ...(formData.adults || []).filter((a: any) => a && a.recommendation).map((a: any) => a.recommendation),
                ...(formData.children || []).filter((c: any) => c && c.recommendation).map((c: any) => c.recommendation)
              ];

              // Group by package code and count occurrences
              const packageGroups: { [key: string]: { count: number; name: string; pax: number } } = {};
              selectedPackageCodes.forEach(code => {
                if (!packageGroups[code]) {
                  // Find package details
                  let typeCriteria: 'L' | 'ST' | Array<'L' | 'ST'> = 'L';
                  if (selectedActivities.surfLessons && selectedActivities.surfTours) {
                    typeCriteria = ['L', 'ST'];
                  } else if (selectedActivities.surfTours) {
                    typeCriteria = 'ST';
                  } else {
                    typeCriteria = 'L';
                  }

                  const personCriteria = {
                    level: 'B' as 'B' | 'I' | 'A',
                    age: 'Ad' as 'A' | 'Ad' | 'Ch1' | 'Ch2',
                    type: typeCriteria,
                    bookingDate: reservationDays[0]?.date,
                    personCount: (adultCount || 0) + (childrenCount || 0), // Total people for minimum pax validation
                  };

                  const recommendedPackages = getRecommendedPackagesForPerson(personCriteria);
                  const packageInfo = recommendedPackages.find((pkg: any) => pkg.code === code);
                  
                  if (packageInfo) {
                    packageGroups[code] = {
                      count: 0,
                      name: packageInfo.name,
                      pax: packageInfo.pax || 1
                    };
                  }
                }
                
                // Only increment if the package group exists
                if (packageGroups[code]) {
                  packageGroups[code].count++;
                }
              });

              // Check validation for each package
              const validationMessages: Array<{ package: string; packageName: string; current: number; required: number; needed: number }> = [];
              
              Object.entries(packageGroups).forEach(([code, group]) => {
                const current = group.count;
                const required = group.pax;
                
                if (current < required || current % required !== 0) {
                  let needed = 0;
                  if (current < required) {
                    needed = required - current;
                  } else {
                    needed = required - (current % required);
                  }
                  
                  validationMessages.push({
                    package: code,
                    packageName: group.name,
                    current,
                    required,
                    needed
                  });
                }
              });

              return validationMessages.length > 0 ? (
                <div className="space-y-3">
                  <h4 className="text-sm font-semibold text-red-800">Package Requirements Not Met</h4>
                  {validationMessages.map((req, index) => (
                    <div key={index} className="bg-red-50 border border-red-200 rounded-lg p-3">
                      <div className="flex items-center gap-2">
                        <svg className="w-5 h-5 text-red-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16.5c-.77.833.192 2.5 1.732 2.5z" />
                        </svg>
                        <div className="text-sm text-red-800">
                          <div className="font-medium">
                            Package "{req.packageName}" needs {req.needed} more people.
                          </div>
                          <div className="mt-1 text-xs text-red-600">
                            Currently {req.current} people selected, but this package requires {req.required} people or multiples of {req.required}.
                          </div>
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              ) : null;
            })()}
          </div>
        )}
      </div>
    </>
  );
}

// Medical Conditions Accordion Component
function MedicalConditionsAccordion({ 
  participant, 
  participantIndex, 
  isChild, 
  formData, 
  setFormData 
}: {
  participant: any;
  participantIndex: number;
  isChild: boolean;
  formData: any;
  setFormData: (data: any) => void;
}) {
  const [isOpen, setIsOpen] = useState(false);

  const medicalConditions = [
    { value: "no_medical", label: "No medical conditions" },
    { value: "heart_condition", label: "Heart condition" },
    { value: "asthma", label: "Asthma" },
    { value: "epilepsy", label: "Epilepsy" },
    { value: "broken_bones", label: "Broken bones" },
    { value: "dislocated_joints", label: "Dislocated joints" },
    { value: "diabetes", label: "Diabetes" },
    { value: "water_contact_lenses", label: "Water contact lenses" },
    { value: "hearing_impairment", label: "Hearing impairment" },
    { value: "pregnant", label: "Pregnant" },
    { value: "other", label: "Other" }
  ];

  const handleMedicalChange = (condition: string, checked: boolean) => {
    let newMedical: string[];
    
    if (condition === "no_medical") {
      // If "No medical conditions" is checked, clear all others
      newMedical = checked ? ["no_medical"] : [];
    } else {
      // If any other condition is checked, remove "no_medical" and add/remove the condition
      const currentMedical = participant.medical || [];
      if (checked) {
        newMedical = [...currentMedical.filter((c: string) => c !== "no_medical"), condition];
      } else {
        newMedical = currentMedical.filter((c: string) => c !== condition);
      }
    }
    
    if (isChild) {
      const newChildren = [...formData.children];
      newChildren[participantIndex] = { ...newChildren[participantIndex], medical: newMedical };
      setFormData({ ...formData, children: newChildren });
    } else {
      const newAdults = [...formData.adults];
      newAdults[participantIndex] = { ...newAdults[participantIndex], medical: newMedical };
      setFormData({ ...formData, adults: newAdults });
    }
  };

  const handleMedicalOtherChange = (value: string) => {
    if (isChild) {
      const newChildren = [...formData.children];
      newChildren[participantIndex] = { ...newChildren[participantIndex], medical_other: value };
      setFormData({ ...formData, children: newChildren });
    } else {
      const newAdults = [...formData.adults];
      newAdults[participantIndex] = { ...newAdults[participantIndex], medical_other: value };
      setFormData({ ...formData, adults: newAdults });
    }
  };

  return (
    <div className="border border-gray-200 rounded-lg">
      {/* Accordion Header */}
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="w-full p-4 bg-gray-50 rounded-lg flex items-center justify-between hover:bg-gray-100 transition-colors"
      >
        <div className="flex items-center gap-3">
          <div className="w-6 h-6 bg-blue-500 rounded flex items-center justify-center">
            <svg className="w-4 h-4 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
          </div>
          <div className="text-left">
            <h5 className="text-base font-semibold text-gray-900">Medical Conditions</h5>
            <p className="text-sm text-gray-600">Please select any medical conditions that apply</p>
          </div>
        </div>
        <svg 
          className={`w-5 h-5 text-gray-500 transition-transform ${isOpen ? 'rotate-180' : ''}`} 
          fill="none" 
          stroke="currentColor" 
          viewBox="0 0 24 24"
        >
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
        </svg>
      </button>

      {/* Accordion Content */}
      {isOpen && (
        <div className="p-4 border-t border-gray-200">
          {/* Medical Conditions Grid */}
          <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-3">
            {medicalConditions.map((cond) => (
              <label key={cond.value} className="flex items-center p-3 border border-gray-200 rounded-lg hover:bg-gray-50 transition-colors cursor-pointer">
                <input
                  type="checkbox"
                  checked={(participant.medical || []).includes(cond.value)}
                  onChange={(e) => handleMedicalChange(cond.value, e.target.checked)}
                  className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                />
                <span className="ml-3 text-sm text-gray-700">{cond.label}</span>
              </label>
            ))}
          </div>

          {/* Custom medical condition input */}
          {(participant.medical?.includes("other") || false) && (
            <div className="mt-4 p-3 bg-blue-50 rounded-lg border border-blue-200">
              <label className="text-sm font-medium text-blue-800 mb-2 block">Please specify:</label>
              <Input
                className="w-full h-10 text-sm border-blue-300 focus:border-blue-500 focus:ring-blue-500"
                value={participant.medical_other || ""}
                onChange={(e) => handleMedicalOtherChange(e.target.value)}
                placeholder="Describe your medical condition..."
              />
            </div>
          )}
        </div>
      )}
    </div>
  );
}