"use client"

import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Calendar, Users, CreditCard, MapPin, User, X } from 'lucide-react';

interface BookingConfirmationModalProps {
  isOpen: boolean;
  onClose: () => void;
  onConfirm: () => void;
  onCheckStatus?: () => void;
  bookingData: {
    bookingNumber: string;
    date: string;
    duration: string;
    participants: string;
    totalAmount: string;
    customerName: string;
    hotel: string;
    hotelAddress: string;
    bookingName: string;
    hotelTransfer: string;
    packages: Array<{
      name: string;
      price: number;
    }>;
  };
  isLoading?: boolean;
  isCheckingStatus?: boolean;
  showCheckStatusButton?: boolean;
  preventClose?: boolean;
  isMonitoringPayPal?: boolean;
}

export default function BookingConfirmationModal({
  isOpen,
  onClose,
  onConfirm,
  onCheckStatus,
  bookingData,
  isLoading = false,
  isCheckingStatus = false,
  showCheckStatusButton = false,
  preventClose = false,
  isMonitoringPayPal = false
}: BookingConfirmationModalProps) {
  return (
    <Dialog 
      open={isOpen} 
      onOpenChange={onClose}
    >
      <DialogContent 
        className="sm:max-w-2xl max-h-[90vh] flex flex-col"
        onPointerDownOutside={preventClose ? (e) => e.preventDefault() : undefined}
        onEscapeKeyDown={preventClose ? (e) => e.preventDefault() : undefined}
      >
        <DialogHeader className="flex-shrink-0">
          <DialogTitle className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <CreditCard className="h-5 w-5 text-blue-600" />
              Confirm PayPal Payment
            </div>
            {!preventClose && (
              <Button
                variant="ghost"
                size="sm"
                onClick={onClose}
                className="h-8 w-8 p-0 hover:bg-gray-100"
              >
                <X className="h-4 w-4" />
              </Button>
            )}
          </DialogTitle>
        </DialogHeader>

        <div className="flex-1 overflow-y-auto space-y-6 pr-2">
          {/* Booking Summary */}
          <div className="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl p-6 border-2 border-blue-200">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
              <Calendar className="h-5 w-5 text-blue-600" />
              Booking Summary
            </h3>
            
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              <div className="space-y-3">
                <div>
                  <span className="text-sm font-medium text-gray-600">Booking Number:</span>
                  <p className="text-lg font-bold text-gray-900">{bookingData.bookingNumber}</p>
                </div>
                
                <div>
                  <span className="text-sm font-medium text-gray-600">Date:</span>
                  <p className="text-gray-900">{bookingData.date}</p>
                </div>
                
                <div>
                  <span className="text-sm font-medium text-gray-600">Duration:</span>
                  <p className="text-gray-900">{bookingData.duration}</p>
                </div>
                
                <div>
                  <span className="text-sm font-medium text-gray-600">Participants:</span>
                  <p className="text-gray-900">{bookingData.participants}</p>
                </div>
              </div>
              
              <div className="space-y-3">
                <div>
                  <span className="text-sm font-medium text-gray-600">Customer Name:</span>
                  <p className="text-gray-900">{bookingData.customerName}</p>
                </div>
                
                {bookingData.hotelTransfer === "Yes" && (
                  <>
                    <div>
                      <span className="text-sm font-medium text-gray-600">Hotel:</span>
                      <p className="text-gray-900">{bookingData.hotel}</p>
                    </div>
                    
                    <div>
                      <span className="text-sm font-medium text-gray-600">Hotel Address:</span>
                      <p className="text-gray-900">{bookingData.hotelAddress}</p>
                    </div>
                    
                    <div>
                      <span className="text-sm font-medium text-gray-600">Hotel Transfer:</span>
                      <p className="text-gray-900">{bookingData.hotelTransfer}</p>
                    </div>
                  </>
                )}
              </div>
            </div>
          </div>

          {/* Packages */}
          <div className="bg-white rounded-xl border border-gray-200 p-6">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
              <Users className="h-5 w-5 text-green-600" />
              Selected Packages
            </h3>
            
            <div className="space-y-3">
              {bookingData.packages && bookingData.packages.length > 0 ? (
                bookingData.packages.map((pkg, index) => (
                  <div key={index} className="flex justify-between items-center p-3 bg-gray-50 rounded-lg">
                    <span className="font-medium text-gray-900">{pkg.name}</span>
                    <span className="text-green-600 font-semibold">${pkg.price.toFixed(2)}</span>
                  </div>
                ))
              ) : (
                <div className="p-3 text-gray-500 text-center bg-gray-50 rounded-lg">
                  No packages selected.
                </div>
              )}
            </div>
          </div>

          {/* Payment Information */}
          <div className="bg-gradient-to-r from-green-50 to-emerald-50 rounded-xl p-6 border-2 border-green-200">
            <h3 className="text-lg font-semibold text-gray-900 mb-4 flex items-center gap-2">
              <CreditCard className="h-5 w-5 text-green-600" />
              Payment Information
            </h3>
            
            <div className="flex justify-between items-center">
              <div>
                <span className="text-sm font-medium text-gray-600">Payment Method:</span>
                <p className="text-gray-900 font-semibold">PayPal</p>
              </div>
              
              <div className="text-right">
                <span className="text-sm font-medium text-gray-600">Total Amount:</span>
                <p className="text-2xl font-bold text-green-600">{bookingData.totalAmount}</p>
              </div>
            </div>
          </div>

          {/* Important Notice */}
          <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
            <div className="flex items-start gap-3">
              <div className="w-5 h-5 bg-yellow-500 rounded-full flex items-center justify-center flex-shrink-0 mt-0.5">
                <span className="text-white text-xs">!</span>
              </div>
              <div>
                <h4 className="text-sm font-semibold text-yellow-800 mb-1">Important Notice</h4>
                <ul className="text-sm text-yellow-700 space-y-1">
                  <li>• Please review all booking details before proceeding</li>
                  <li>• Payment will be processed securely through PayPal</li>
                  <li>• You will be redirected to PayPal to complete the payment</li>
                  <li>• Booking will be confirmed after successful payment</li>
                </ul>
              </div>
            </div>
          </div>

        </div>

        {/* Monitoring Status Indicator */}
        {isMonitoringPayPal && (
          <div className="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-4">
            <div className="flex items-center gap-3">
              <div className="w-6 h-6 bg-blue-500 rounded-full flex items-center justify-center">
                <div className="w-3 h-3 bg-white rounded-full animate-pulse"></div>
              </div>
              <div>
                <p className="text-sm font-medium text-blue-800">Monitoring Payment Status</p>
                <p className="text-xs text-blue-600">We're automatically checking your payment status. Please complete payment in the PayPal tab.</p>
              </div>
            </div>
          </div>
        )}

        {/* Action Buttons - Fixed at bottom */}
        <div className="flex-shrink-0 pt-4 border-t border-gray-200">
          {showCheckStatusButton ? (
            <div className="flex gap-3">
              <Button
                onClick={onConfirm}
                disabled={isLoading}
                className="flex-1 bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white"
              >
                {isLoading ? (
                  <div className="flex items-center gap-2">
                    <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                    <span>Processing...</span>
                  </div>
                ) : (
                  <div className="flex items-center gap-2">
                    <CreditCard className="h-4 w-4" />
                    <span>Proceed to PayPal</span>
                  </div>
                )}
              </Button>
              
              <Button
                onClick={onCheckStatus}
                disabled={isCheckingStatus || isMonitoringPayPal}
                variant="outline"
                className="flex-1"
              >
                {isCheckingStatus ? (
                  <div className="flex items-center gap-2">
                    <div className="w-4 h-4 border-2 border-gray-400 border-t-transparent rounded-full animate-spin"></div>
                    <span>Checking...</span>
                  </div>
                ) : isMonitoringPayPal ? (
                  <div className="flex items-center gap-2">
                    <div className="w-4 h-4 border-2 border-blue-400 border-t-transparent rounded-full animate-spin"></div>
                    <span>Auto-checking...</span>
                  </div>
                ) : (
                  <div className="flex items-center gap-2">
                    <CreditCard className="h-4 w-4" />
                    <span>Check Payment Status</span>
                  </div>
                )}
              </Button>
            </div>
          ) : (
            <Button
              onClick={onConfirm}
              disabled={isLoading}
              className="w-full bg-gradient-to-r from-blue-600 to-indigo-600 hover:from-blue-700 hover:to-indigo-700 text-white"
            >
              {isLoading ? (
                <div className="flex items-center gap-2">
                  <div className="w-4 h-4 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
                  <span>Processing...</span>
                </div>
              ) : (
                <div className="flex items-center gap-2">
                  <CreditCard className="h-4 w-4" />
                  <span>Proceed to PayPal</span>
                </div>
              )}
            </Button>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
} 