import { NextRequest, NextResponse } from "next/server";

// Simple test endpoint that always returns COMPLETED for testing
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const bookingNo = searchParams.get('booking_no');

    console.log("🧪 Test Payment Status Request:", bookingNo);

    if (!bookingNo) {
      return NextResponse.json(
        { error: "Booking number is required" },
        { status: 400 }
      );
    }

    // For testing: Simulate realistic payment completion
    const isRecentBooking = bookingNo.includes('BK-20251029');
    
    if (isRecentBooking) {
      // Simulate payment completion based on booking number pattern
      // This makes it more predictable for testing
      const bookingSuffix = bookingNo.split('-').pop(); // Get last part (e.g., "0001")
      const bookingNumber = parseInt(bookingSuffix || '0');
      
      // Simulate payment completion after a shorter, more realistic time
      // Different booking numbers complete at different times for testing
      const baseTime = bookingNumber * 5000; // 5 seconds per booking number
      const minTimeForSuccess = Math.min(baseTime, 15000); // Max 15 seconds
      
      const bookingTime = new Date().getTime();
      const timeSinceBooking = Date.now() - bookingTime;
      
      const isPaymentSuccessful = timeSinceBooking > minTimeForSuccess;
      
      if (isPaymentSuccessful) {
        console.log(`✅ Test: Returning COMPLETED for booking: ${bookingNo} (${Math.round(timeSinceBooking/1000)}s elapsed)`);
        return NextResponse.json({
          success: true,
          message: "Payment status retrieved (test mode)",
          data: {
            booking_no: bookingNo,
            status: "COMPLETED",
            payment_status: "COMPLETED",
            message: "Payment completed successfully (test mode)"
          },
        });
      } else {
        console.log(`⏳ Test: Returning PENDING for booking: ${bookingNo} (${Math.round(timeSinceBooking/1000)}s elapsed, need ${Math.round(minTimeForSuccess/1000)}s)`);
        return NextResponse.json({
          success: true,
          message: "Payment status retrieved (test mode)",
          data: {
            booking_no: bookingNo,
            status: "PENDING",
            payment_status: "PENDING",
            message: "Payment is being processed (test mode)"
          },
        });
      }
    } else {
      console.log("⏳ Test: Returning PENDING for booking:", bookingNo);
      return NextResponse.json({
        success: true,
        message: "Payment status retrieved (test mode)",
        data: {
          booking_no: bookingNo,
          status: "PENDING",
          payment_status: "PENDING",
          message: "Payment is being processed (test mode)"
        },
      });
    }
  } catch (error) {
    console.error("❌ Test Payment Status Error:", error);
    return NextResponse.json(
      { error: "Internal server error" },
      { status: 500 }
    );
  }
}
