import { NextRequest, NextResponse } from "next/server";

// Quick test endpoint for immediate payment success
export async function GET(request: NextRequest) {
  try {
    const { searchParams } = new URL(request.url);
    const bookingNo = searchParams.get('booking_no');

    console.log("⚡ Quick Test Payment Status Request:", bookingNo);

    if (!bookingNo) {
      return NextResponse.json(
        { error: "Booking number is required" },
        { status: 400 }
      );
    }

    // For responsive testing: Detect payment completion more quickly
    const isRecentBooking = bookingNo.includes('BK-20251029');
    
    if (isRecentBooking) {
      // Simulate responsive payment detection
      // Check if enough time has passed for client to complete payment
      const bookingTime = new Date().getTime();
      const timeSinceBooking = Date.now() - bookingTime;
      const minTimeForSuccess = 8000; // 8 seconds - faster detection
      
      // Also check if this is a recent booking that should be completed
      const bookingSuffix = bookingNo.split('-').pop();
      const bookingNumber = parseInt(bookingSuffix || '0');
      
      // For testing: Complete payment faster for recent bookings
      const isPaymentSuccessful = timeSinceBooking > minTimeForSuccess || bookingNumber >= 1;
      
      if (isPaymentSuccessful) {
        console.log(`✅ Responsive Test: Payment completed for booking: ${bookingNo} (${Math.round(timeSinceBooking/1000)}s elapsed)`);
        return NextResponse.json({
          success: true,
          message: "Payment status retrieved (responsive test mode)",
          data: {
            booking_no: bookingNo,
            status: "COMPLETED",
            payment_status: "COMPLETED",
            message: "Payment completed successfully (responsive test mode)"
          },
        });
      } else {
        console.log(`⏳ Responsive Test: Payment pending for booking: ${bookingNo} (${Math.round(timeSinceBooking/1000)}s elapsed, need ${Math.round(minTimeForSuccess/1000)}s)`);
        return NextResponse.json({
          success: true,
          message: "Payment status retrieved (responsive test mode)",
          data: {
            booking_no: bookingNo,
            status: "PENDING",
            payment_status: "PENDING",
            message: "Payment is being processed (responsive test mode)"
          },
        });
      }
    } else {
      console.log("⚡ Quick Test: Returning PENDING for booking:", bookingNo);
      return NextResponse.json({
        success: true,
        message: "Payment status retrieved (quick test mode)",
        data: {
          booking_no: bookingNo,
          status: "PENDING",
          payment_status: "PENDING",
          message: "Payment is being processed (quick test mode)"
        },
      });
    }
  } catch (error) {
    console.error("❌ Quick Test Payment Status Error:", error);
    return NextResponse.json(
      { error: "Internal server error" },
      { status: 500 }
    );
  }
}
