# PayPal Webhook Setup Guide

## 🔧 Current Issue

The PayPal webhook for payment approval is not working. This guide will help you set it up properly.

## 📋 What I've Created

### 1. Webhook Endpoint

- **File**: `app/api/paypal/webhook/route.ts`
- **URL**: `https://yourdomain.com/api/paypal/webhook`
- **Purpose**: Receives webhook notifications from PayPal

### 2. Payment Status Update Endpoint

- **File**: `app/api/paypal/update-payment-status/route.ts`
- **URL**: `https://yourdomain.com/api/paypal/update-payment-status`
- **Purpose**: Updates payment status in your backend

## 🚀 Setup Steps

### Step 1: Configure PayPal Webhook in PayPal Developer Dashboard

1. **Login to PayPal Developer Dashboard**

   - Go to https://developer.paypal.com/
   - Login with your PayPal account

2. **Select Your App**

   - Go to "My Apps & Credentials"
   - Select your app (or create one if needed)

3. **Configure Webhook**

   - Go to "Webhooks" section
   - Click "Add Webhook"
   - **Webhook URL**: `https://yourdomain.com/api/paypal/webhook`
   - **Event Types to Subscribe**:
     - `PAYMENT.CAPTURE.COMPLETED` ✅
     - `PAYMENT.CAPTURE.DENIED` ✅
     - `PAYMENT.CAPTURE.PENDING` ✅
     - `PAYMENT.CAPTURE.REFUNDED` ✅

4. **Save Webhook Configuration**
   - Copy the Webhook ID (you'll need this for verification)

### Step 2: Update Your Backend API

Make sure your backend has these endpoints:

#### Required Backend Endpoints:

1. **Update Payment Status**

   ```
   POST /paypal/update-payment-status
   ```

   **Request Body**:

   ```json
   {
     "booking_no": "string",
     "payment_id": "string",
     "status": "APPROVED|PENDING|DENIED|REFUNDED",
     "paypal_status": "string",
     "amount": "number",
     "currency": "string",
     "webhook_data": "object"
   }
   ```

2. **Get Payment Status**
   ```
   GET /paypal/booking-payment-status/{bookingNo}
   ```
   **Response**:
   ```json
   {
     "success": true,
     "data": {
       "status": "APPROVED|PENDING|DENIED",
       "payment_id": "string",
       "amount": "number",
       "currency": "string"
     }
   }
   ```

### Step 3: Update PayPal Payment Creation

When creating PayPal payments, make sure to include the booking number in the `custom_id` field:

```javascript
// In your backend when creating PayPal payment
const paypalPayment = {
  // ... other payment details
  custom_id: bookingNumber, // This is crucial for webhook processing
  description: `Booking: ${bookingNumber}`, // Optional but helpful
};
```

### Step 4: Test the Webhook

1. **Use PayPal Webhook Simulator**

   - Go to PayPal Developer Dashboard
   - Navigate to your webhook
   - Click "Test Webhook"
   - Send test events

2. **Check Logs**
   - Monitor your application logs
   - Look for webhook processing messages
   - Check if payment status is being updated

## 🔍 Debugging

### Check Webhook Logs

Look for these log messages in your application:

```
🔔 PayPal Webhook Received:
✅ Payment Completed Webhook Processing
✅ Payment status updated successfully
```

### Common Issues

1. **Webhook Not Receiving Events**

   - Check if webhook URL is accessible
   - Verify webhook is enabled in PayPal dashboard
   - Check if SSL certificate is valid

2. **Payment Status Not Updating**

   - Verify backend API endpoint exists
   - Check API authentication token
   - Verify database update logic

3. **Booking Number Not Found**
   - Ensure `custom_id` is set when creating PayPal payment
   - Check webhook data structure

## 🧪 Testing

### Manual Test

1. Create a PayPal payment
2. Complete payment on PayPal
3. Check if webhook receives the event
4. Verify payment status is updated
5. Check if frontend shows correct status

### Webhook Test Events

Use PayPal's webhook simulator to test different scenarios:

- Payment completed
- Payment denied
- Payment pending
- Payment refunded

## 📝 Environment Variables

Make sure these are set in your `.env.local`:

```env
NEXT_PUBLIC_API_URL=https://your-backend-api.com
NEXT_PUBLIC_API_TOKEN=your-api-token
```

## 🔒 Security Notes

1. **Webhook Verification**: The current implementation skips webhook verification for simplicity. In production, implement proper PayPal webhook signature verification.

2. **Rate Limiting**: Consider adding rate limiting to prevent abuse.

3. **Error Handling**: The webhook includes comprehensive error handling and logging.

## 📞 Support

If you're still having issues:

1. Check the application logs for webhook processing
2. Verify PayPal webhook configuration
3. Test with PayPal's webhook simulator
4. Ensure backend API endpoints are working

The webhook should now automatically update payment status when PayPal processes payments!
